(ns raid.base
  (:require [raid.async :as async]
            [raid.envelop :as envelop]
            [raid.logging :as log]
            [raid.recipes :as recipes])
  (:gen-class))

(defn call-handlers [f handlers]
  (when (seq handlers)
    (f (first handlers))
    (recur f (rest handlers))))

(defn close-socket [socket]
  (future
    (.close socket)
    true))

(defn on-msg [io payload handler]
  (let [{:keys [ok? msg]} (envelop/unpack payload)]
    (if ok?
      (let [action (get-in msg [:header :action])
            handler (or (recipes/actions action) handler)]
        (log/info (format "MSG RECEIVED: %s" msg))
        (async/add-callback handler io msg)))))

(defmulti socket-closed?
  (fn [this]
    (:type this)))
(defmethod socket-closed? :udp [this]
  (.isClosed (:socket this)))
(defmethod socket-closed? :tcp [this]
  (not (.isOpen (:socket this))))

(defn push-msg [this msg sender]
  (if (socket-closed? this)
    false
    (let [payload (envelop/pack msg)]
      (if payload
        (do
          (sender payload)
          (log/info (format "MSG SENT: %s" msg)))
        false))))

(defn read-data [this reader]
  (let [{:keys [ok? data ex]} (reader)]
    (if ok?
      (do
        (call-handlers #(async/add-callback on-msg this data %) @(:msg-handlers this))
        (recur this reader))
      (if-not (socket-closed? this)
        (call-handlers #(async/add-callback % this ex) @(:ex-handlers this))))))
