(ns raid.core
  (:import java.net.InetAddress)
  (:require [raid.base :as base]
            [raid.resources :refer [sha-hash uuid4]]
            [raid.tcp :as tcp]
            [raid.udp :as udp])
  (:gen-class))

(def default-timeout (atom 5000))

(def ^:private conns (atom {}))

(def ^:private msgs (atom []))

(defrecord Connection [socket host port type name msg-handlers ex-handlers run etags])

(defn- add-conn [conn name]
  (swap! conns assoc name conn))

(defn add-msg-handler [this handler]
  {:pre [(instance? Connection this)]}
  (if (fn? handler)
    (do
      (swap! (:msg-handlers this) conj handler)
      true)
    false))

(defn add-ex-handler [this handler]
  {:pre [(instance? Connection this)]}
  (if (fn? handler)
    (do
      (swap! (:ex-handlers this) conj handler)
      true)
    false))

(defn- def-handler [this msg]
  (let [etag (get-in msg [:header :etag])
        fut (get (deref (:etags this)) etag)]
    (if fut
      (deliver fut msg))))

(defmulti start
  (fn [this]
    {:pre [(instance? Connection this)]}
    (:type this)))
(defmethod start :tcp [this]
  (tcp/start-read this))
(defmethod start :udp [this]
  (udp/start-read this))

(defn- gen-name []
  (sha-hash "SHA-256" (format "socket-%s" (uuid4))))

(defn- basic-connect [host port socket type msg-handler ex-handler]
  (let [name (gen-name)]
    (doto (map->Connection {:socket socket
                            :host host
                            :port port
                            :type type
                            :msg-handlers (atom [def-handler])
                            :name name
                            :ex-handlers (atom nil)
                            :run (atom false)
                            :etags (atom {})})
      (add-msg-handler msg-handler)
      (add-ex-handler ex-handler)
      start
      (add-conn name))))

(defn cancel [this]
  {:pre [(instance? Connection this)]}
  (base/close-socket (:socket this)))

(defn get-socket [this]
  {:pre [(instance? Connection this)]}
  (:socket this))

(defmulti push
  (fn [this msg]
    {:pre [(instance? Connection this)]}
    (:type this)))
(defmethod push :tcp [this msg]
  (tcp/push-msg this msg))
(defmethod push :udp [this msg]
  (udp/push-msg this msg))

(defn- push-msg [headers action etag body]
  {:header (merge headers
                  {:action action
                   :etag etag})
   :body body})

(defn reach-host? [host]
  {:pre [(string? host) (not-empty host)]}
  (.isReachable (InetAddress/getByName host) @default-timeout))

(defn- request-response [fut timeout]
  (deref fut timeout :timeout))

(defn- respond-promise [this etag fut timeout]
  (let [resp (request-response fut timeout)]
    (swap! (:etags this) dissoc etag)
    (if (= resp :timeout)
      {:ok? false :err :timeout}
      {:ok? true :resp (deref fut)})))

(defn request [this action body & [headers {:keys [timeout]}]]
  {:pre [(instance? Connection this) (string? action) (not-empty action)
         (or (nil? timeout) (and (integer? timeout) (pos? timeout)))]}
  (if-not (deref (:run this))
    (start this))
  (let [etag (uuid4)
        fut (promise)]
    (swap! (:etags this) assoc etag fut)
    (push this (push-msg headers action etag body))
    (respond-promise this etag fut (or timeout @default-timeout))))

(defn set-timeout [value]
  {:pre [(pos? value)]}
  (reset! default-timeout value))

(defn tcp-connect
  "Open a TCP connection on a RAID server"
  [host port & [msg-handler ex-handler]]
  (basic-connect host
                 port
                 (tcp/create-socket host port)
                 :tcp
                 msg-handler
                 ex-handler))

(defn udp-connect
  "Open a UDP connection on a RAID server"
  [host port & [msg-handler ex-handler]]
  (basic-connect host
                 port
                 (udp/create-socket port)
                 :udp
                 msg-handler
                 ex-handler))
