(ns raid.core
  (:require [raid.base :as base]
            [raid.tcp :as tcp]
            [raid.udp :as udp])
  (:gen-class))

(defprotocol Connection
  (add-msg-handler [this handler])
  (add-ex-handler [this handler])
  (cancel [this])
  (get-socket [this])
  (push [this msg])
  (start [this]))

(defrecord TCP [socket host port msg-handlers ex-handlers run])

(defrecord UDP [socket host port msg-handlers ex-handlers run])

(extend-protocol Connection
  TCP
  (add-msg-handler [this handler] (base/add-msg-handler this handler))
  (add-ex-handler [this handler] (base/add-ex-handler this handler))
  (cancel [this] (base/close-socket (:socket this)))
  (get-socket [this] (:socket this))
  (push [this msg] (tcp/push-msg this msg))
  (start [this] (tcp/start-read this))
  UDP
  (add-msg-handler [this handler] (base/add-msg-handler this handler))
  (add-ex-handler [this handler] (base/add-ex-handler this handler))
  (cancel [this] (base/close-socket (:socket this)))
  (get-socket [this] (:socket this))
  (push [this msg] (udp/push-msg this msg))
  (start [this] (udp/start-read this)))

(defn tcp-connect
  "Open a TCP connection on a RAID server"
  [host port & [msg-handler ex-handler]]
  (doto (map->TCP {:socket (tcp/create-socket host port)
                   :host host
                   :port port
                   :msg-handlers (atom nil)
                   :ex-handlers (atom nil)
                   :run (atom false)})
    (add-msg-handler msg-handler)
    (add-ex-handler ex-handler)
    start))


(defn udp-connect
  "Open a UDP connection on a RAID server"
  [host port & [msg-handler ex-handler]]
  (doto (map->UDP {:socket (udp/create-socket port)
                   :host host
                   :port port
                   :msg-handlers (atom nil)
                   :ex-handlers (atom nil)
                   :run (atom false)})
    (add-msg-handler msg-handler)
    (add-ex-handler ex-handler)
    start))
