(ns njord.soap
  (:require [augustus.async :as async]
            [augustus.future :as afuture]
            [cheshire.core :as json]
            [clojure.data.codec.base64 :as b64]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.queue :as queue]
            [njord.common :refer [sha256 uuid4]]
            [njord.kernel :as kernel])
  (:import com.novemberain.langohr.Connection)
  (:gen-class))

(defrecord NjordSoap [ch uid])

(def ^:const options
  {:exchange {:headers "njord_soap_headers"
              :topic "njord_soap"}
   :rk "njord.soap.execute"})

(defn- decode [msg]
  (json/parse-string (new String ^bytes (b64/decode (.getBytes ^String msg))) true))

(defn- encode [msg]
  (b64/encode (.getBytes ^String (json/generate-string msg))))

(defn- gen-etag [^NjordSoap io]
  (sha256 (format "njord-soap[%s:%s]" (:uid io) (uuid4))))

(defn- queue-bind [^NjordSoap io]
  (let [exchange (get-in options [:exchange :headers])
        arguments {"requester" (:uid io)
                   "njord-soap-response" true
                   "x-match" "all"}]
    (queue/bind (:ch io) (kernel/get-queue) exchange {:arguments arguments})))

(defn execute-async [^NjordSoap io request & [cb]]
  {:pre [(instance? NjordSoap io) (or (nil? cb) (fn? cb))]}
  (let [etag (gen-etag (:uuid io))
        exchange (get-in options [:exchange :topic])
        rk (:rk options)
        fut (afuture/new-future)
        body (encode request)
        headers {:headers {"requester" (:uid io)
                           "etag" etag
                           "service" "execute"}}]
    (if cb
      (afuture/add-future-callback fut cb))
    (kernel/add-promise etag #(fut (decode %)))
    (basic/publish (:ch io) exchange rk body headers)
    fut))

(defn execute [^NjordSoap io request]
  {:pre [(instance? NjordSoap io)]}
  (deref (execute-async io request)))

(defn start [^Connection conn]
  {:pre [(instance? Connection conn)]}
  (let [io (map->NjordSoap
            {:ch (channel/open conn)
             :uid (uuid4)})]
    (kernel/start conn)
    (async/add-callback queue-bind io)
    io))
