(ns njord.http
  (:require [augustus.async :as async]
            [augustus.future :as afuture]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.queue :as queue]
            [njord.common :refer [sha256 uuid4]]
            [njord.kernel :as kernel]
            [utcode.core :as utcode])
  (:import com.novemberain.langohr.Connection)
  (:gen-class))

(defrecord NjordHttp [ch uid])

(def ^:const options
  {:exchange {:headers "njord_http_headers"
              :topic "njord_http"}
   :rk "njord.http.request"})

(defn- gen-etag [^NjordHttp io]
  (sha256 (format "njord-http[%s:%s]" (:uid io) (uuid4))))

(defn- queue-bind [^NjordHttp io]
  (let [exchange (get-in options [:exchange :headers])
        arguments {"requester" (:uid io)
                   "njord-http-response" true
                   "x-match" "all"}]
    (queue/bind (:ch io) (kernel/get-queue) exchange {:arguments arguments})))

(defn request-async [^NjordHttp io request & [cb]]
  {:pre [(instance? NjordHttp io) (or (nil? cb) (fn? cb))]}
  (let [etag (gen-etag (:uuid io))
        exchange (get-in options [:exchange :topic])
        rk (:rk options)
        fut (afuture/new-future)
        body (utcode/encode request)
        headers {:headers {"requester" (:uid io)
                           "etag" etag
                           "service" "fetch"}}]
    (if cb
      (afuture/add-future-callback fut cb))
    (kernel/add-promise etag #(fut (utcode/decode %)))
    (basic/publish (:ch io) exchange rk body headers)
    fut))

(defn request [^NjordHttp io request]
  {:pre [(instance? NjordHttp io)]}
  (deref (request-async io request)))

(defn start [^Connection conn]
  {:pre [(instance? Connection conn)]}
  (let [io (map->NjordHttp
            {:ch (channel/open conn)
             :uid (uuid4)})]
    (kernel/start conn)
    (async/add-callback queue-bind io)
    io))
