(ns njord.http
  (:import com.novemberain.langohr.Connection)
  (:require [clojure.tools.logging :as log]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.consumers :as consumers]
            [langohr.core :as rmq]
            [langohr.queue :as queue]
            [njord.http.common :refer [thread-call sha1 uuid4]]
            [utcode.core :as utcode])
  (:gen-class))

(defprotocol NjordHttpProt
  (fetch [this request]))

(declare do-request)

(defrecord NjordHttp [requests conn chan-out uuid]
  NjordHttpProt
  (fetch [this request] (do-request this request)))

(def ^:const options
  {:exchange
   {:headers "njord_http_headers"
    :topic "njord_http"}
   :queue "njord_http_%s"
   :rk "njord.http.request"})

(defn- gen-etag [^NjordHttp this]
  (sha1 (format "njord-http[%s:%s]" (:uuid this) (uuid4))))

(defn- call-promise [requests etag payload]
  (let [promises @requests]
    (when (contains? promises etag)
      ((get promises etag) payload)
      (swap! requests dissoc etag))))

(defn- on-msg [^NjordHttp this chan {:keys [headers delivery-tag]} ^bytes body]
  (if (contains? headers "etag")
    (let [etag (str (get headers "etag"))]
      (call-promise (:requests this) etag (String. body))))
  (basic/ack chan delivery-tag))

(defn- queue-declare [^NjordHttp this]
  (let [queue (format (options :queue) (gen-etag (:uuid this)))
        arguments {"requester" (:uuid this)
                   "njord-http-response" true
                   "x-match" "all"}]
    (doto (channel/open (:conn this))
      (basic/qos 10)
      (queue/declare queue {:durable false :exclusive true})
      (queue/bind queue (-> options :exchange :headers) {:arguments arguments})
      (consumers/subscribe queue (fn [& args] (thread-call (apply on-msg this args)))))))

(defn- do-request [^NjordHttp this request]
  (let [etag (gen-etag (:uuid this))
        fut (promise)]
    (thread-call
     (swap! (:requests this) assoc etag #(deliver fut (utcode/decode %)))
     (basic/publish @(:chan-out this)
                    (-> options :exchange :topic)
                    (options :rk)
                    (utcode/encode request)
                    {:headers {"requester" (:uuid this)
                               "etag" etag
                               "service" "fetch"}}))
    fut))

(defn init [^Connection conn]
  (assert (instance? Connection conn))
  (let [this (NjordHttp. (atom {}) conn (future (channel/open conn)) (uuid4))]
    (thread-call (queue-declare this))
    (log/info "Njord Http ready")
    this))
