(ns augustus.future
  (:require [augustus.pools :as pools])
  (:import augustus.future.exceptions.AlreadyCompleted
           [clojure.lang IBlockingDeref IDeref IFn IPending]
           [java.util.concurrent CancellationException CountDownLatch Future TimeoutException TimeUnit])
  (:gen-class))

(definterface IFuture
  (^Boolean addCallback [^clojure.lang.IFn callback])
  (^augustus.future.IFuture setException [^Exception exception])
  (^augustus.future.IFuture setResult [^Object result]))

(defn add-future-callback [^IFuture fut callback]
  (.addCallback fut callback))

(defn cancel [^Future fut]
  (.cancel fut false))

(defn set-exception [^IFuture fut ex]
  (.setException fut ex))

(defn set-result [^IFuture fut result]
  (.setResult fut result))

(defn- call-handler [^IFuture fut callbacks]
  (when (seq callbacks)
    (let [callback (first callbacks)]
      (.execute pools/main-pool (fn [] (callback fut))))
    (recur fut (rest callbacks))))

(defn new-future
  ([]
   (let [d (new CountDownLatch 1)
         ex (atom nil)
         v (atom nil)
         cb (atom [])]
     (reify
       IDeref
       (deref [this]
         (.get this))
       IBlockingDeref
       (deref [this timeout-ms timeout-val]
         (try
           (.get this timeout-ms TimeUnit/MILLISECONDS)
           (catch TimeoutException e
             timeout-val)))
       IPending
       (isRealized [this]
         (.isDone this))
       Future
       (get [_]
         (.await d)
         (if (deref ex)
           (throw (deref ex))
           (deref v)))
       (get [this timeout unit]
         (if (.await d timeout unit)
           (if (deref ex)
             (throw (deref ex))
             (deref v))
           (throw (new TimeoutException))))
       (isCancelled [_]
         (instance? CancellationException (deref ex)))
       (isDone [_]
         (zero? (.getCount d)))
       (cancel [this _]
         (if (realized? this)
           false
           (do
             (.setException this (new CancellationException))
             true)))
       IFn
       (invoke [this v]
         (if (instance? Throwable v)
           (.setException this v)
           (.setResult this v)))
       IFuture
       (addCallback [this callback]
         (if (realized? this)
           (call-handler this [callback])
           (swap! cb conj callback))
         true)
       (setException [this exception]
         (if (realized? this)
           (throw (new AlreadyCompleted)))
         (reset! ex exception)
         (.countDown d)
         (call-handler this (deref cb))
         this)
       (setResult [this result]
         (if (realized? this)
           (throw (new AlreadyCompleted)))
         (reset! v result)
         (.countDown d)
         (call-handler this (deref cb))
         this))))
  ([callback]
   (doto (new-future)
     (add-future-callback callback))))
