(ns augustus.future
  (:require [augustus.pools :as pools])
  (:import augustus.future.exceptions.AlreadyCompleted
           [clojure.lang IBlockingDeref IDeref IFn IPending]
           [java.util.concurrent Future CompletableFuture ExecutionException TimeoutException TimeUnit])
  (:gen-class))

(definterface IFuture
  (^Boolean addCallback [^clojure.lang.IFn callback])
  (^augustus.future.IFuture setException [^Exception exception])
  (^augustus.future.IFuture setResult [^Object result]))

(defn add-future-callback [^IFuture fut callback]
  (.addCallback fut callback))

(defn set-exception [^IFuture fut ex]
  (.setException fut ex))

(defn set-result [^IFuture fut result]
  (.setResult fut result))

(defn- call-handler [^IFuture fut callbacks]
  (when (seq callbacks)
    (let [callback (first callbacks)]
      (.execute pools/main-pool (fn [] (callback fut))))
    (recur fut (rest callbacks))))

(defn new-future
  ([]
   (let [fut (new CompletableFuture)
         cb (atom [])]
     (reify
       IDeref
       (deref [this] (.get this))
       IBlockingDeref
       (deref [_ timeout-ms timeout-val]
         (try
           (.get fut timeout-ms TimeUnit/MILLISECONDS)
           (catch TimeoutException e
             timeout-val)))
       IPending
       (isRealized [_] (.isDone fut))
       Future
       (get [_]
         (try
           (.get fut)
           (catch ExecutionException e
             (throw (.getCause e)))))
       (get [_ timeout unit]
         (try
           (.get fut timeout unit)
           (catch ExecutionException e
             (throw (.getCause e)))))
       (isCancelled [_] (.isCancelled fut))
       (isDone [_] (.isDone fut))
       (cancel [_ interrupt?] (.cancel fut interrupt?))
       IFn
       (invoke [this v] (if (instance? Exception v)
                          (.setException this v)
                          (.setResult this v)))
       IFuture
       (addCallback [this callback]
         (if (.isDone fut)
           (call-handler this [callback])
           (swap! cb conj callback))
         true)
       (setException [this exception]
         (if (.isDone fut)
           (throw (new AlreadyCompleted)))
         (.completeExceptionally fut exception)
         (call-handler this (deref cb))
         this)
       (setResult [this result]
         (if (.isDone fut)
           (throw (new AlreadyCompleted)))
         (.complete fut result)
         (call-handler this (deref cb))
         this))))
  ([callback]
   (doto (new-future)
     (add-future-callback callback))))
