(ns sci.impl.destructure
  "Destructure function, adapted from Clojure and ClojureScript."
  {:no-doc true}
  (:refer-clojure :exclude [destructure])
  (:require [sci.impl.utils :refer [gensym* mark-eval-call]]))

(defn destructure* [bindings]
  (let [bents (partition 2 bindings)
        pb (fn pb [bvec b v]
             (let [pvec
                   (fn [bvec b val]
                     (let [gvec (gensym* "vec__")
                           gseq (gensym* "seq__")
                           gfirst (gensym* "first__")
                           has-rest (some #{'&} b)]
                       (loop [ret (let [ret (conj bvec gvec val)]
                                    (if has-rest
                                      (conj ret gseq (mark-eval-call (list seq gvec)))
                                      ret))
                              n 0
                              bs b
                              seen-rest? false]
                         (if (seq bs)
                           (let [firstb (first bs)]
                             (cond
                               (= firstb '&) (recur (pb ret (second bs) gseq)
                                                    n
                                                    (nnext bs)
                                                    true)
                               (= firstb :as) (pb ret (second bs) gvec)
                               :else (if seen-rest?
                                       (throw #?(:clj (new Exception "Unsupported binding form, only :as can follow & parameter")
                                                 :cljs (new js/Error "Unsupported binding form, only :as can follow & parameter")))
                                       (recur (pb (if has-rest
                                                    (conj ret
                                                          gfirst (mark-eval-call `(~first ~gseq))
                                                          gseq (mark-eval-call `(~next ~gseq)))
                                                    ret)
                                                  firstb
                                                  (if has-rest
                                                    gfirst
                                                    (mark-eval-call (list nth gvec n nil))))
                                              (inc n)
                                              (next bs)
                                              seen-rest?))))
                           ret))))
                   pmap
                   (fn [bvec b v]
                     (let [gmap (gensym* "map__")
                           defaults (:or b)]
                       (loop [ret (-> bvec (conj gmap) (conj v)
                                      (conj gmap) (conj (mark-eval-call
                                                         `(if ~(mark-eval-call `(~seq? ~gmap))
                                                            ~(mark-eval-call `(~apply ~hash-map ~gmap))
                                                            ~gmap)))
                                      ((fn [ret]
                                         (if (:as b)
                                           (conj ret (:as b) gmap)
                                           ret))))
                              bes (let [transforms
                                        (reduce
                                         (fn [transforms mk]
                                           (if (keyword? mk)
                                             (let [mkns (namespace mk)
                                                   mkn (name mk)]
                                               (cond (= mkn "keys")
                                                     (assoc transforms mk
                                                            #(keyword (or mkns (namespace %))
                                                                      (name %)))
                                                     (= mkn "syms")
                                                     (assoc transforms mk
                                                            #(symbol (or mkns
                                                                         (namespace %))
                                                                     (name %)))
                                                     (= mkn "strs") (assoc transforms mk str)
                                                     :else transforms))
                                             transforms))
                                         {}
                                         (keys b))]
                                    (reduce
                                     (fn [bes entry]
                                       (reduce #(assoc %1 %2 ((val entry) %2))
                                               (dissoc bes (key entry))
                                               ((key entry) bes)))
                                     (dissoc b :as :or)
                                     transforms))]
                         (if (seq bes)
                           (let [bb (key (first bes))
                                 bk (val (first bes))
                                 local (if #?(:clj  (instance? clojure.lang.Named bb)
                                              :cljs (implements? INamed bb))
                                         (with-meta (symbol nil (name bb)) (meta bb))
                                         bb)
                                 ;; if bk has metadata (:line, :column, etc)
                                 ;; then it's a binding that needs eval
                                 bk (if-not (meta bk)
                                      (list 'quote bk)
                                      bk)
                                 bv (if-let [default (get defaults local)]
                                      (mark-eval-call (list get gmap bk default))
                                      (mark-eval-call (list get gmap bk)))]
                             (recur
                              (if (or (keyword? bb) (symbol? bb)) ;(ident? bb)
                                (-> ret (conj local bv))
                                (pb ret bb bv))
                              (next bes)))
                           ret))))]
               (cond
                 (symbol? b) (-> bvec (conj (if (namespace b)
                                              (symbol (name b)) b)) (conj v))
                 (keyword? b) (-> bvec (conj (symbol (name b))) (conj v))
                 (vector? b) (pvec bvec b v)
                 (map? b) (pmap bvec b v)
                 :else (throw
                        #?(:clj (new Exception (str "Unsupported binding form: " b))
                           :cljs (new js/Error (str "Unsupported binding form: " b)))))))
        process-entry (fn [bvec b] (pb bvec (first b) (second b)))]
    (if (every? symbol? (map first bents))
      bindings
      (if-let [kwbs (seq (filter #(keyword? (first %)) bents))]
        (throw
         #?(:clj (new Exception (str "Unsupported binding key: " (ffirst kwbs)))
            :cljs (new js/Error (str "Unsupported binding key: " (ffirst kwbs)))))
        (reduce process-entry [] bents)))))

(defn destructure [b]
  (destructure* b))
