(ns boot.task.built-in
  (:require
   [clojure.java.io    :as io]
   [clojure.set        :as set]
   [clojure.pprint     :as pprint]
   [clojure.string     :as string]
   [clojure.stacktrace :as trace]
   [boot.pod           :as pod]
   [boot.core          :as core]
   [boot.main          :as main]
   [boot.conch         :as conch]
   [boot.file          :as file]
   [boot.util          :as util]
   [boot.table.core    :as table]))

(defn- first-line [s] (when s (first (string/split s #"\n"))))
(defn- not-blank? [s] (when-not (string/blank? s) s))
(defn- now        [ ] (System/currentTimeMillis))
(defn- ms->s      [x] (double (/ x 1000)))

(defmacro ^:private print-time [ok fail expr]
  `(let [start# (now)]
     (try
       (let [end# (do ~expr (ms->s (- (now) start#)))]
         (printf ~ok end#))
       (catch Throwable e#
         (let [time#  (ms->s (- (now) start#))
               trace# (with-out-str (trace/print-cause-trace e#))]
           (println (format ~fail trace# time#)))))))

(defn- tasks-table [tasks]
  (let [get-task  #(-> % :name str)
        get-desc  #(-> % :doc first-line)]
    (->> tasks (map (fn [x] ["" (get-task x) (get-desc x)])))))

(defn- set-title [[[_ & xs] & zs] title] (into [(into [title] xs)] zs))

(defn- version-str []
  (format "Boot Version:  %s\nDocumentation: %s"
    (core/get-env :boot-version) "http://github.com/tailrecursion/boot"))

(defn- available-tasks [sym]
  (let [base  {nil (the-ns sym)}
        task? #(:boot.core/task %)
        addrf #(if-not (seq %1) %2 (symbol %1 (str %2)))
        proc  (fn [a [k v]] (assoc (meta v) :name (addrf a k) :var v))
        pubs  (fn [[k v]] (map proc (repeat (str k)) (ns-publics v)))]
    (->>
      (concat
        (->> sym ns-refers (map proc (repeat nil)))
        (->> sym ns-aliases (into base) (mapcat pubs)))
      (filter task?) (sort-by :name))))

(defn- parse-map-entry [m k v]
  (let [[k' v'] (string/split v #"=" 2)]
    ((fnil assoc-in {}) m [k (keyword k')] v')))

(defn- parse-set-member [m k v]
  (update-in m [k] (fnil conj #{}) v))

(defn- parse-vector-item [m k v]
  (update-in m [k] (fnil conj []) v))

(def ^:private ^:dynamic *sh-dir* nil)

(defn sh [& args]
  (let [opts (into [:redirect-err true] (when *sh-dir* [:dir *sh-dir*]))
        proc (apply conch/proc (concat args opts))]
    (future (conch/stream-to-out proc :out))
    #(.waitFor (:process proc))))

(defn- pp-str [form]
  (with-out-str (pprint/pprint form)))

(defn- ns-requires [sym]
  (let [ns   (the-ns sym)
        core (the-ns 'clojure.core)
        sift #(remove (partial contains? #{nil ns core}) (set %))]
    (-> (->> sym ns-aliases vals)
      (concat (->> sym ns-map vals (map (comp :ns meta))))
      sift)))

(def ^:private bgs
  "List of tasks running in other threads that will need to be cleaned up before
  boot can exit."
  (atom ()))

;; cleanup background tasks on shutdown
(-> (Runtime/getRuntime)
  (.addShutdownHook (Thread. #(doseq [job @bgs] (future-cancel job)))))

;; Task constructors ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn repl-server
  [opts]
  (core/with-pre-wrap
    (try (require 'clojure.tools.nrepl.server)
         (catch Throwable _
           (pod/add-dependencies
             (assoc (core/get-env)
               :dependencies '[[org.clojure/tools.nrepl "0.2.4"]]))))
    (require 'boot.repl-server)
    ((resolve 'boot.repl-server/start-server) opts)))

(defn repl-client
  [opts]
  (core/with-pre-wrap
    (pod/with-worker
      (require '[boot.repl-client :as client])
      (client/client ~(-> opts (assoc :color (not (:no-color opts))))))))

(defn pom*
  [& {:keys [project version description license scm url] :as opts}]
  (let [opts  (assoc opts :dependencies (:dependencies (core/get-env)))
        done? #(->> (core/tgt-files) (core/by-name ["pom.xml"]) first)]
    (core/with-pre-wrap
      (when-not (done?)
        (when-not (and project version)
          (throw (Exception. "need project and version to create pom.xml")))
        (let [tgt (core/mktgtdir! ::pom-tgt)]
          (pod/with-worker
            (require '[boot.pom :as pom])
            (pom/spit-pom! ~(.getPath tgt) '~opts)))))))

(defn jar*
  [& {:keys [no-source manifest main filter-src filter-jar]
      :or   {filter-src identity filter-jar identity}}]
  (core/consume-src! (partial core/by-name ["pom.xml" "pom.properties"]))
  (core/with-pre-wrap
    (let [jarfile (->> (core/tgt-files) (core/by-ext [".jar"]) first)]
      (when-not jarfile
        (let [pomprop (->> (core/tgt-files) (core/by-name ["pom.properties"]) first)]
          (when-not pomprop
            (throw (Exception. "can't find pom.properties file")))
          (let [{:keys [project version]} (pod/pom-properties-map pomprop)
                jarname (util/jarname project version)
                tgt     (core/mktgtdir! ::jar-tgt)
                jarfile (io/file tgt jarname)
                index   (->> (if no-source (core/tgt-files) (core/src-files))
                          (map (juxt core/relative-path (memfn getPath))) (into {}))
                ks      (set (filter-jar (keys index)))
                vs      (set (filter-src (vals index)))
                keep?   #(and (contains? ks %1) (contains? vs %2))
                paths   (->> index (reduce-kv #(if-not (keep? %2 %3) %1 (assoc %1 %2 %3)) {}))]
            (pod/with-worker
              (require '[boot.jar :as jar])
              (jar/spit-jar! ~(.getPath jarfile) ~(vec paths) '~manifest '~main))))))))

(defn uberjar*
  [& {:keys [scopes] :or {scopes #{"compile" "runtime" "provided"}}}]
  (core/with-pre-wrap
    (let [jarfile (->> (core/tgt-files) (core/by-ext [".jar"]) first)]
      (when-not jarfile (throw (Exception. "can't find jar file")))
      (let [uberfile (->> (core/tgt-files) (core/by-ext [".uber.jar"]) first)]
        (when-not uberfile
          (let [ubername (str (.replaceAll (.getName jarfile) "[.]jar$" "") ".uber.jar")
                tgt      (core/mktgtdir! ::uberjar-tgt)
                uberfile (io/file tgt ubername)
                scope?   #(contains? scopes (:scope (util/dep-as-map %)))
                env      (-> (core/get-env)
                           (update-in [:dependencies] (partial filter scope?)))]
            (pod/with-worker
              (require
                '[boot.jar :as jar]
                '[boot.aether :as aether])
              (->> (aether/jar-entries-in-dep-order '~env)
                (jar/uber-jar! ~(.getPath jarfile) ~(.getPath uberfile))))))))))

(defn help
  "Print help and usage info for boot, plus a list of names and descriptions
  for tasks that are available from the command line."
  []
  (core/with-pre-wrap
    (let [tasks (available-tasks 'boot.user)
          opts  (->> main/cli-opts (mapv (fn [[x y z]] ["" (str x " " y) z])))]
      (printf "%s\n\n" (version-str))
      (printf "%s\n"
        (-> [[""       ""]
             ["Usage:" "boot OPTS <task> TASK_OPTS <task> TASK_OPTS ..."]]
          (table/table :style :none)
          with-out-str))
      (printf "%s\nDo `boot <task> -h` to see usage info and TASK_OPTS for <task>.\n"
        (-> [["" "" ""]]
          (into (set-title (conj opts ["" "" ""]) "OPTS:"))
          (into (set-title (tasks-table tasks) "Tasks:"))
          (table/table :style :none)
          with-out-str)))))

(defn once
  "Evaluate the given `task` only once, then pass through."
  [task]
  (let [ran? (atom false)
        run? (partial compare-and-set! ran? @ran?)]
    (fn [continue]
      (let [task (task continue)]
        #(continue ((if (run? true) task identity) %))))))

(defn bg-task
  "Run the given task once, in a separate, background thread."
  [task]
  (once
    (core/with-pre-wrap
      (swap! bgs conj (future ((task identity) core/*event*))))))

(defn auto
  "Run every `msec` (default 200) milliseconds."
  [& [msec]]
  (fn [continue]
    (fn [event]
      (continue event)
      (Thread/sleep (or msec 200))
      (recur (core/make-event event)))))

(defn files-changed?
  [& [type quiet?]]
  (let [dirs      (remove core/tmpfile? (core/get-env :src-paths)) 
        watchers  (map file/make-watcher dirs)
        since     (atom 0)]
    (fn [continue]
      (fn [event]
        (let [clean #(assoc %2 %1 (set (remove core/ignored? (get %2 %1))))
              info  (->> (map #(%) watchers)
                      (reduce (partial merge-with set/union))
                      (clean :time)
                      (clean :hash))]
          (if-let [mods (->> (or type :time) (get info) seq)]
            (do
              (let [path   (file/path (first mods))
                    ok-v   "\033[34m↳ Elapsed time: %6.3f sec ›\033[33m 00:00:00 \033[0m"
                    ok-q   "Elapsed time: %6.3f sec\n"
                    fail-v "\n\033[31m%s\033[0m\n\033[34m↳ Elapsed time: %6.3f sec ›\033[33m 00:00:00 \033[0m"
                    fail-q "\n%s\nElapsed time: %6.3f sec\n"
                    ok     (if quiet? ok-q ok-v)
                    fail   (if quiet? fail-q fail-v)]
                (when (not= 0 @since) (println)) 
                (reset! since (:time event))
                (print-time ok fail (continue (assoc event :watch info)))
                (flush)))
            (let [diff  (long (/ (- (:time event) @since) 1000))
                  pad   (apply str (repeat 9 "\b"))
                  s     (mod diff 60)
                  m     (mod (long (/ diff 60)) 60)
                  h     (mod (long (/ diff 3600)) 24)]
              (core/sync!)
              (when-not quiet?
                (printf "\033[33m%s%02d:%02d:%02d \033[0m" pad h m s)
                (flush)))))))))

;; Tasks ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(core/deftask debug
  "Print the boot environment map."
  []
  (core/with-pre-wrap
    (print (pp-str (core/get-env)))))

(core/deftask wait
  "Wait before calling the next handler."
  
  [["-t" "--time MSEC" "Wait MSEC milliseconds, or forever if zero." :default 0 :parse-fn read-string]]

  (core/with-pre-wrap
    (if (zero? time) @(promise) (Thread/sleep time))))

(core/deftask watch
  "Call the next handler whenever source files change."
  
  [["-t" "--time MSEC" "Set polling interval to MSEC milliseconds." :default 200 :parse-fn read-string]
   ["-q" "--quiet"     "Suppress updating timer and ANSI color output."]]

  (comp (auto time) (files-changed? :time quiet)))

(core/deftask syncdir
  "Copy/sync files between directories.

  The `in-dir` directories will be overlayed on the `out-dir` directory. Empty 
  directories are ignored. Similar to `rsync --delete` in a Unix system."
  
  [["-i" "--in-dirs DIR"  "Add DIR to set of source directories." :assoc-fn parse-set-member]
   ["-o" "--out-dir DIR" "Set destination directory to DIR."]]

  (core/add-sync! out-dir in-dirs)
  identity)

(core/deftask repl
  "Start a REPL session for the current project.

  The #'boot.repl-server/*default-middleware* dynamic var holds a vector of the
  default REPL middleware to be included. You may modify this in your build.boot
  file by calling set! or rebinding the var."

  [["-s" "--server"    "Start only server."]
   ["-c" "--client"    "Start only client."]
   ["-b" "--bind ADDR" "Bind server to ADDR." :default "0.0.0.0"]
   ["-H" "--host HOST" "Connect client to HOST." :default "127.0.0.1"]
   ["-p" "--port PORT" "Connect to/listen on port PORT." :parse-fn read-string]
   ["-C" "--no-color"  "Disable ANSI color output."]]

  (cond
    server (repl-server (select-keys *opts* [:bind :port :init-ns :middleware]))
    client (repl-client (select-keys *opts* [:host :port :no-color :history]))
    :else  (comp
             (apply repl :server true (mapcat identity *opts*))
             (apply repl :client true (mapcat identity *opts*)))))

(core/deftask dep-tree
  "Print the project's dependency graph."
  []
  (core/with-pre-wrap
    (pod/with-worker
      (require '[boot.aether :as aether])
      (aether/print-dep-tree '~(core/get-env)))))

(core/deftask pom
  "Write the project's pom.xml file."

  [["-p" "--project PROJECT"   "The project groupId/artifactId." :parse-fn read-string]
   ["-v" "--version VERSION"   "The project version."]
   ["-d" "--description DESC"  "A description of the project."]
   ["-u" "--url URL"           "The URL for the project homepage."]
   ["-l" "--license KEY=VAL"   "Add KEY => VAL to license map (KEY one of name, url)." :assoc-fn parse-map-entry]
   ["-s" "--scm KEY=VAL"       "Add KEY => VAL to scm map (KEY one of url, tag)." :assoc-fn parse-map-entry]]

  (->> *opts* (mapcat identity) (apply pom*)))

(core/deftask jar
  "Build a jar file for the project."
  
  [["-S" "--no-source"        "Exclude source files from jar."]
   ["-M" "--manifest KEY=VAL" "Add KEY => VAL to jar manifest map." :assoc-fn parse-map-entry]
   ["-m" "--main MAIN"        "The namespace containing the -main function." :parse-fn read-string]
   ["-s" "--filter-src FUNC"  "Function to pass source file paths through for filtering." :parse-fn (comp eval read-string)]
   ["-j" "--filter-jar FUNC"  "Function to pass jar paths through for filtering." :parse-fn (comp eval read-string)]]

  (comp (pom) (apply jar* (mapcat identity *opts*))))

(core/deftask uberjar
  "Build project jar with dependencies included.

  By default, dependencies with the following scopes will be included in the
  uber jar file: compile, runtime, and provided."
  
  [["-x" "--exclude-scope SCOPE" "Exclude dependencies with this scope from uberjar." :assoc-fn parse-set-member]]

  (let [dfl-scopes #{"compile" "runtime" "provided"}
        scopes     (set/difference dfl-scopes exclude-scope)
        *opts*     (-> *opts* (dissoc :exclude-scope) (assoc :scopes scopes))]
    (comp
      (jar)
      (apply uberjar* (mapcat identity *opts*)))))

(core/deftask install
  "Install project jar to local Maven repository."
  
  [["-f" "--file PATH" "Jar file to install."]]

  (comp
    (if file identity (jar))
    (core/with-pre-wrap
      (let [jarfile (or (and file (io/file file))
                      (->> (core/tgt-files) (core/by-ext [".jar"]) first))]
        (when-not jarfile
          (throw (Exception. "can't find jar file")))
        (pod/with-worker
          (require
            '[boot.pod :as pod]
            '[boot.aether :as aether])
          (pod/info "Installing %s...\n" ~(.getName jarfile))
          (aether/install ~(.getPath jarfile))
          nil)))))

(core/deftask push
  "Push project jar to Clojars."
  
  [["-f" "--file PATH" "Jar file to push."]]

  (comp
    (if file identity (jar))
    (core/with-pre-wrap
      (let [jarfile (if file
                      (io/file file)
                      (->> (core/tgt-files) (core/by-ext [".jar"]) first))]
        (when-not jarfile (throw (Exception. "can't find jar file")))
        (let [tmp (core/mktmpdir! ::push-tmp)
              pom (doto (io/file tmp "pom.xml") (spit (pod/pom-xml jarfile)))]
          ((sh "scp" (.getPath jarfile) (.getPath pom) "clojars@clojars.org:")))))))
