(ns boot.task.built-in
  (:require
   [clojure.java.io    :as io]
   [clojure.set        :as set]
   [clojure.pprint     :as pprint]
   [clojure.string     :as string]
   [clojure.stacktrace :as trace]
   [boot.pod           :as pod]
   [boot.core          :as core]
   [boot.main          :as main]
   [boot.conch         :as conch]
   [boot.file          :as file]
   [boot.util          :as util]
   [boot.table.core    :as table]))

(defn- first-line [s] (when s (first (string/split s #"\n"))))
(defn- not-blank? [s] (when-not (string/blank? s) s))
(defn- now        [ ] (System/currentTimeMillis))
(defn- ms->s      [x] (double (/ x 1000)))

(defmacro ^:private print-time [ok fail expr]
  `(let [start# (now)]
     (try
       (let [end# (do ~expr (ms->s (- (now) start#)))]
         (printf ~ok end#))
       (catch Throwable e#
         (let [time#  (ms->s (- (now) start#))
               trace# (with-out-str (trace/print-cause-trace e#))]
           (println (format ~fail trace# time#)))))))

(defn- tasks-table [tasks]
  (let [get-task  #(-> % :name str)
        get-desc  #(-> % :doc first-line)]
    (->> tasks (map (fn [x] ["" (get-task x) (get-desc x)])))))

(defn- set-title [[[_ & xs] & zs] title] (into [(into [title] xs)] zs))

(defn- version-str []
  (format "Boot Version:  %s\nDocumentation: %s"
    (core/get-env :boot-version) "http://github.com/tailrecursion/boot"))

(defn- available-tasks [sym]
  (let [base  {nil (the-ns sym)}
        task? #(:boot.core/task %)
        addrf #(if-not (seq %1) %2 (symbol %1 (str %2)))
        proc  (fn [a [k v]] (assoc (meta v) :name (addrf a k) :var v))
        pubs  (fn [[k v]] (map proc (repeat (str k)) (ns-publics v)))]
    (->>
      (concat
        (->> sym ns-refers (map proc (repeat nil)))
        (->> sym ns-aliases (into base) (mapcat pubs)))
      (filter task?) (sort-by :name))))

(def ^:private ^:dynamic *sh-dir* nil)

(defn sh [& args]
  (let [opts (into [:redirect-err true] (when *sh-dir* [:dir *sh-dir*]))
        proc (apply conch/proc (concat args opts))]
    (future (conch/stream-to-out proc :out))
    #(.waitFor (:process proc))))

(defn- pp-str [form]
  (with-out-str (pprint/pprint form)))

(defn- ns-requires [sym]
  (let [ns   (the-ns sym)
        core (the-ns 'clojure.core)
        sift #(remove (partial contains? #{nil ns core}) (set %))]
    (-> (->> sym ns-aliases vals)
      (concat (->> sym ns-map vals (map (comp :ns meta))))
      sift)))

;; ## Core Tasks
;;
;; These tasks are included in boot core and are listed by the `help` task.

(core/deftask debug
  "Print the boot environment map.

  Multiple `keys` specify a path, similar to `clojure.core/get-in`."
  [& keys]
  (core/with-pre-wrap
    (let [e (core/get-env)]
      (print (pp-str (if-not (seq keys) e (get-in e keys)))))))

(core/deftask help
  "Print help and usage info for a task.
  
  Help and usage info is derived from metadata on the task var. The task must be
  required into the boot script namespace for this medatada to be found."
  ([]
     (core/with-pre-wrap
       (let [tasks (available-tasks 'boot.user)
             opts  (->> main/cli-opts (mapv (fn [[x y z]] ["" (str x " " y) z])))]
         (printf "%s\n\n" (version-str))
         (printf "%s\n"
           (-> [[""       ""]
                ["Usage:" "boot OPTS task ..."]
                [""       "boot OPTS [task arg arg] ..."]
                [""       "boot OPTS [help task]"]]
             (table/table :style :none)
             with-out-str))
         (printf "%s\n"
           (-> [["" "" ""]]
             (into (set-title (conj opts ["" "" ""]) "OPTS:"))
             (into (set-title (tasks-table tasks) "Tasks:"))
             (table/table :style :none)
             with-out-str)))))
  ([task]
     (core/with-pre-wrap
       (let [task* (->> (available-tasks 'boot.user)
                     (map :var) (filter #(= task (var-get %))) first)
             {args :arglists doc :doc task :name} (meta task*)]
         (when-not task*
           (throw (Exception. "Not a valid task")))
         (printf "%s\n\n" (version-str))
         (printf "%s\n%s\n  %s\n\n" task args doc)))))

(defn- generate-lein-project-file!
  [& {:keys [keep-project] :or {:keep-project true}}]
  (let [pfile (io/file "project.clj")
        pname (or (core/get-env :project) 'boot-project)
        pvers (or (core/get-env :version) "0.1.0-SNAPSHOT")
        prop  #(when-let [x (core/get-env %2)] [%1 x])
        head  (list* 'defproject pname pvers
                     (concat
                      (prop :url :url)
                      (prop :license :license)
                      (prop :description :description)
                      [:dependencies (core/get-env :dependencies)
                       :source-paths (vec (core/get-env :src-paths))]))
        proj  (pp-str (concat head (mapcat identity (core/get-env :lein))))]
    (if-not keep-project (.deleteOnExit pfile))
    (spit pfile proj)))

(core/deftask lein-generate
  "Generate a leiningen `project.clj` file.

  This task generates a leiningen `project.clj` file based on the boot
  environment configuration, including project name and version (generated
  if not present), dependencies, and source paths. Additional keys may be added
  to the generated `project.clj` file by specifying a `:lein` key in the boot
  environment whose value is a map of keys-value pairs to add to `project.clj`."
  []
  (core/with-pre-wrap
    (generate-lein-project-file! :keep-project true)))

(core/deftask lein
  "Run a leiningen task with a generated `project.clj`.

  This task generates a leiningen `project.clj` file based on the boot
  environment configuration, including project name and version (generated
  if not present), dependencies, and source paths. Additional keys may be added
  to the generated `project.clj` file by specifying a `:lein` key in the boot
  environment whose value is a map of keys-value pairs to add to `project.clj`.

  Once the `project.clj` file has been generated, the specified lein task is
  then run. Note that leiningen is run in another process. This task cannot be
  used to run interactive lein tasks (yet) because stdin is not currently piped
  to leiningen."
  [& args]
  (core/with-pre-wrap
    (generate-lein-project-file! :keep-project true)
    ((apply sh "lein" (map str args)))))

(def ^:private bgs
  "List of tasks running in other threads that will need to be cleaned up before
  boot can exit."
  (atom ()))

;; cleanup background tasks on shutdown
(-> (Runtime/getRuntime)
  (.addShutdownHook (Thread. #(doseq [job @bgs] (future-cancel job)))))

(core/deftask once
  "Evaluate the given `task` only once.

  Subsequent evaluations will pass through."
  [task]
  (let [ran? (atom false)
        run? (partial compare-and-set! ran? @ran?)]
    (fn [continue]
      (let [task (task continue)]
        #(continue ((if (run? true) task identity) %))))))

(core/deftask wait
  "Wait for msec milliseconds before calling its continuation.

  If called with no argument waits forever."
  [& [msec]]
  (core/with-pre-wrap
    (if-not msec @(promise) (Thread/sleep msec))))

(core/deftask background
  "Run the given `task` once, in a separate, background thread."
  [task]
  (once
    (core/with-pre-wrap
      (swap! bgs conj (future ((task identity) core/*event*))))))

(defn auto
  "Run every `msec` (default 200) milliseconds."
  [& [msec]]
  (fn [continue]
    (fn [event]
      (continue event)
      (Thread/sleep (or msec 200))
      (recur (core/make-event event)))))

(defn files-changed?
  [& [type quiet?]]
  (let [dirs      (remove core/tmpfile? (core/get-env :src-paths)) 
        watchers  (map file/make-watcher dirs)
        since     (atom 0)]
    (fn [continue]
      (fn [event]
        (let [clean #(assoc %2 %1 (set (remove core/ignored? (get %2 %1))))
              info  (->> (map #(%) watchers)
                      (reduce (partial merge-with set/union))
                      (clean :time)
                      (clean :hash))]
          (if-let [mods (->> (or type :time) (get info) seq)]
            (do
              (let [path   (file/path (first mods))
                    ok-v   "\033[34m↳ Elapsed time: %6.3f sec ›\033[33m 00:00:00 \033[0m"
                    ok-q   "Elapsed time: %6.3f sec\n"
                    fail-v "\n\033[31m%s\033[0m\n\033[34m↳ Elapsed time: %6.3f sec ›\033[33m 00:00:00 \033[0m"
                    fail-q "\n%s\nElapsed time: %6.3f sec\n"
                    ok     (if quiet? ok-q ok-v)
                    fail   (if quiet? fail-q fail-v)]
                (when (not= 0 @since) (println)) 
                (reset! since (:time event))
                (print-time ok fail (continue (assoc event :watch info)))
                (flush)))
            (let [diff  (long (/ (- (:time event) @since) 1000))
                  pad   (apply str (repeat 9 "\b"))
                  s     (mod diff 60)
                  m     (mod (long (/ diff 60)) 60)
                  h     (mod (long (/ diff 3600)) 24)]
              (core/sync!)
              (when-not quiet?
                (printf "\033[33m%s%02d:%02d:%02d \033[0m" pad h m s)
                (flush)))))))))

(core/deftask watch
  "Watch `:src-paths` and call its continuation when files change.

  Options:

    :type    Specifies how changes to files are detected and can be either 
             :time or :hash (default :time).
    :msec    Specifies the polling interval in milliseconds (default 200).
    :quiet   When set to true no ANSI colors or updating timer are printed. "
  [& {:keys [type msec quiet] :or {type :time msec 200 quiet false}}]
  (comp (auto msec) (files-changed? type quiet)))

(core/deftask syncdir
  "Copy/sync files between directories.

  The files in `from-dirs` will be overlayed on the `to-dir` directory. Empty 
  directories are ignored."
  [to-dir & from-dirs]
  (core/add-sync! to-dir from-dirs)
  identity)

(defn- repl-server
  [& opts]
  (core/with-pre-wrap
    (try (require 'clojure.tools.nrepl.server)
         (catch Throwable _
           (pod/add-dependencies
             (assoc (core/get-env)
               :dependencies '[[org.clojure/tools.nrepl "0.2.4"]]))))
    (require 'boot.repl-server)
    ((resolve 'boot.repl-server/start-server) (apply hash-map opts))))

(defn- repl-client
  [& opts]
  (core/with-pre-wrap
    (pod/with-worker
      (require '[boot.repl-client :as client])
      (client/client ~(apply hash-map opts)))))

(core/deftask repl
  "Start a REPL session for the current project.

  Options: :start         Either :server or :client (both).
           :bind          Address to bind the REPL server to (0.0.0.0).
           :host          Address/hostname to connect REPL client to (127.0.0.1).
           :port          Port for server or client to listen on or connect to
                          (0 for server, reads \".nrepl-port\" for client).
           :color         Should the client use colored output (true)?
           :init-ns       Keyword of namespace to load in REPL (:boot.user).
           :middleware    Vector of nREPL middleware vars ([]).
           :history-file  Path to REPL client's history file (\".nrepl-history\")."
  [& {:keys [start bind host port color init-ns middleware history-file]
      :or {color true} :as opts}]
  (case start
    :server (repl-server :bind bind :port port :init-ns init-ns :middleware middleware)
    :client (repl-client :host host :port port :color color :history-file history-file)
    (comp
      (apply repl :start :server (mapcat identity opts))
      (apply repl :start :client (mapcat identity opts)))))

(core/deftask dep-tree
  "Print the project's dependency graph."
  []
  (core/with-pre-wrap
    (pod/with-worker
      (require '[boot.aether :as aether])
      (aether/print-dep-tree '~(core/get-env)))))

(core/deftask pom
  "Write the project's pom.xml file."
  [& {:keys [project version] :as opts}]
  (let [opts  (assoc opts :dependencies (:dependencies (core/get-env)))
        done? #(->> (core/tgt-files) (core/by-name ["pom.xml"]) first)]
    (when-not (and project version)
      (throw (Exception. "need project and version to create pom.xml")))
    (core/with-pre-wrap
      (when-not (done?)
        (let [tgt (core/mktgtdir! ::pom-tgt)]
          (pod/with-worker
            (require '[boot.pom :as pom])
            (pom/spit-pom! ~(.getPath tgt) '~opts)))))))

(core/deftask jar
  "Build a jar file for the project.

  Options: :exclude-src   If true no source files will be included in the jar.
           :manifest      A map of string keys and values to add to jar manifest.
           :main          The namespace symbol in for the main class in manifest.
           :filter-src    The seq of src paths is passed through the given function.
           :filter-jar    The seq of jar paths is passed through the given function."
  [& {:keys [exclude-src manifest main filter-src filter-jar]
      :or   {filter-src identity filter-jar identity}}]
  (core/consume-src! (partial core/by-name ["pom.xml" "pom.properties"]))
  (comp
    (pom)
    (core/with-pre-wrap
      (let [jarfile (->> (core/tgt-files) (core/by-ext [".jar"]) first)]
        (when-not jarfile
          (let [pomprop (->> (core/tgt-files) (core/by-name ["pom.properties"]) first)]
            (when-not pomprop
              (throw (Exception. "can't find pom.properties file")))
            (let [{:keys [project version]} (pod/pom-properties-map pomprop)
                  jarname (util/jarname project version)
                  tgt     (core/mktgtdir! ::jar-tgt)
                  jarfile (io/file tgt jarname)
                  index   (->> (if exclude-src (core/tgt-files) (core/src-files))
                            (map (juxt core/relative-path (memfn getPath))) (into {}))
                  ks      (set (filter-jar (keys index)))
                  vs      (set (filter-src (vals index)))
                  keep?   #(and (contains? ks %1) (contains? vs %2))
                  paths   (->> index (reduce-kv #(if-not (keep? %2 %3) %1 (assoc %1 %2 %3)) {}))]
              (pod/with-worker
                (require '[boot.jar :as jar])
                (jar/spit-jar! ~(.getPath jarfile) ~(vec paths) '~manifest '~main)))))))))

(core/deftask uberjar
  "Build project jar with dependencies included.

  Options: :scopes        Dependencies will be included if their scope is in
                          this set (#{\"compile\", \"provided\", \"runtime\"})."
  [& {:keys [scopes]}]
  (core/consume-src! (partial core/by-re [#"(?<!\.uber)\.jar$"]))
  (comp
    (jar)
    (core/with-pre-wrap
      (let [jarfile (->> (core/tgt-files) (core/by-ext [".jar"]) first)]
        (when-not jarfile (throw (Exception. "can't find jar file")))
        (let [uberfile (->> (core/tgt-files) (core/by-ext [".uber.jar"]) first)]
          (when-not uberfile
            (let [scopes   (or scopes #{"compile" "profided" "runtime"})
                  ubername (str (.replaceAll (.getName jarfile) "[.]jar$" "") ".uber.jar")
                  tgt      (core/mktgtdir! ::uberjar-tgt)
                  uberfile (io/file tgt ubername)
                  scope?   #(contains? scopes (:scope (util/dep-as-map %)))
                  env      (-> (core/get-env)
                             (update-in [:dependencies] (partial filter scope?)))]
              (pod/with-worker
                (require
                  '[boot.jar :as jar]
                  '[boot.aether :as aether])
                (->> (aether/jar-entries-in-dep-order '~env)
                  (jar/uber-jar! ~(.getPath jarfile) ~(.getPath uberfile)))))))))))

(core/deftask install
  "Install project jar to local Maven repository.

  Options: :jar-file      Path to jar file to install."
  [& {:keys [jar-file]}]
  (comp
    (if jar-file identity (jar))
    (core/with-pre-wrap
      (let [jarfile (or (and jar-file (io/file jar-file))
                      (->> (core/tgt-files) (core/by-ext [".jar"]) first))]
        (when-not jarfile
          (throw (Exception. "can't find jar file")))
        (pod/with-worker
          (require
            '[boot.pod :as pod]
            '[boot.aether :as aether])
          (pod/info "Installing %s...\n" ~(.getName jarfile))
          (aether/install ~(.getPath jarfile))
          nil)))))

(core/deftask push
  "Push project jar to Clojars."
  []
  (comp
    (jar)
    (core/with-pre-wrap
      (let [jarfile (->> (core/tgt-files) (core/by-ext [".jar"]) first)]
        (when-not jarfile (throw (Exception. "can't find jar file")))
        (let [tmp (core/mktmpdir! ::push-tmp)
              pom (doto (io/file tmp "pom.xml") (spit (pod/pom-xml jarfile)))]
          ((sh "scp" (.getPath jarfile) (.getPath pom) "clojars@clojars.org:")))))))
