(ns reagent-hooks.core
  "A library designed to emulate react-hooks, but for reagent."
  (:require [reagent.core :as r]
            [reagent.dom :as rdom]
            [reagent.impl.template :as tmpl]
            [reagent.impl.protocols :as rp]))

(defn use-state
  "Returns a atom/setter-fn pair."
  [value]
  {:pre [(any? value)]}
  (let [r (r/atom value)]
    [r #(reset! r %)]))

(deftype ReactRef [^:mutable ^:export current]
  IFn
  (-invoke [this value]
    (vreset! this value))
  IVolatile
  (-vreset! [_ new-state]
    (set! current new-state))
  IDeref
  (-deref [_] current))

(defn use-ref
  "Returns a react-ref similar to React.createRef"
  ([] (use-ref nil))
  ([val]
   (->ReactRef val)))

(defn- use-lifecycle-hook
  "Some dirty work to make hooks work.
  Hooks are attached via monkey-patching.
  This function adds a hook to a given lifecycle callback"
  [hook-name f]
  {:pre [(string? hook-name) (fn? f)]}
  (let [this (r/current-component)
        current-f-raw (aget this hook-name)]
    (aset this hook-name (fn [& args]
                           (f this)
                           (when current-f-raw
                             (.apply current-f-raw this (clj->js args)))))
    nil))

(def use-did-update
  "Adds hook for the componentDidUpdate lifecycle callback.
  Given function F will be called everytime didUpdate will be called."
  (partial use-lifecycle-hook "componentDidUpdate"))

(def use-did-mount
  "Adds hook for the componentDidMount lifecycle callback.
  Given function F will be called everytime didMount will be called. (i.e once per component)"
  (partial use-lifecycle-hook "componentDidMount"))

(def use-will-unmount
  "Adds hook for the componentWillUnmount lifecycle callback.
  Given function F will be called everytime willUnmount will be called. (i.e once on component remove)"
  (partial use-lifecycle-hook "componentWillUnmount"))

(def use-will-catch
  "Adds hook for the componentWillUnmount lifecycle callback.
  Given function F will be called everytime willUnmount will be called. (i.e once on component remove)"
  (partial use-lifecycle-hook "componentWillCatch"))



(defn use-effect
  "Adds a hook for both componentDidMount and componentWillUnmount.
  Given function F will be used on did mount.
  If F returns a function, that function will be called on unmount."
  [f]
  {:pre [(fn? f)]}
  (let [handler (volatile! nil)]
    (use-did-mount (fn []
                     (let [result (f)]
                       (when (fn? result)
                         (vreset! handler result)))))
    (use-will-unmount (fn []
                        (when-let [h @handler]
                          (h))))
    nil))

(defn use-watch
  "Adds a watch on an ATOM.
  Given function F will be called with the latest value of the atom uppon atom change."
  [atom f]
  {:pre [(fn? f)]}
  (use-effect (fn []
                (let [sym (keyword (gensym "watcher"))]
                  (add-watch atom sym (fn [_ _ _ next-value] (f next-value)))
                  #(remove-watch atom sym))))
  nil)

(defn- extend-prototype [type]
  (when (fn? type)
    (doseq [f ["componentDidMount" "componentDidUpdate" "componentDidCatch" "componentWillUnmount"]]
      (when-not (aget (.-prototype type) f)
        (aset (.-prototype type) f identity)))))

(def ^:private compiler-id (gensym))
(def compiler
  (reify rp/Compiler
    (get-id [this] compiler-id)
    (as-element [this x]
      (let [r (tmpl/as-element this x tmpl/reag-element)]
        (extend-prototype (.-type r))
        r))
    (make-element [this argv component jsprops first-child]
      (tmpl/make-element this argv component jsprops first-child))))

(defn render  [comp node]
  (rdom/render comp node compiler))
