# reagent-hooks

[![Clojars Project](https://img.shields.io/clojars/v/boogie666/reagent-hooks.svg)](https://clojars.org/boogie666/reagent-hooks)

React-like hooks for reagent.

## Overview

A libarary to use something similar to react hooks but with reagent.

## Usage

```clojure
(:require [reagent-hooks.core :as h])

;; only for form-2 components
(defn some-component []
 (let [[text set-text!] (h/use-state "Hello")
       text-input-ref (h/use-ref)
       _ (h/use-effect (fn [] (.focus @text-input-ref)
                         (fn [] (.blur @text-input-ref))))]
   (fn []
     [:input {:value @text
              :ref text-input-ref
              :on-change #(set-text! (-> @text-input-ref .-value)})))

```

## API

### The 'Low-Level' api

Described below is the 'low-level' api. These function are the base of the hooks api. (very similar to react-hooks)

### use-state

```clojure
(let [[value setter!] (h/use-state 42)]
  (= 42 @value)
  (setter! 43)
  (= 43 @value))
```

returns a pair of a ratom and a setter function for that ratom


### use-ref

```clojure
(let [ref (h/use-ref)
      some-varialbe (h/use-ref 42)])
```

returns a derefable volatile IFn with a 'current' property.

The returned object is:

* Derefable - for convenience (= @ref (.-current ref))
* Volatile - because it's basically the same as volatile (so vswap! and vreset! work as expected, but settings the .-current property)
* IFn - because reagent :ref prop needs a function that will be called with a ref to the current element. 
    ex: (ref 42) is the same (vreset! ref 42) is the same as (set! (.-current ref) 42)

### use-did-mount

```clojure
  (let [_ (h/use-did-mount (fn [] (println "Printed on component-did-mount")))
```

Adds a callback for the componentDidMount lifecycle hook.

### use-will-unmount

```clojure
  (let [_ (h/use-will-unmount (fn [] (println "Printed on component-will-unmount")))
```

Adds a callback for the componentWillUnmount lifecycle hook.

### use-did-update

```clojure
  (let [_ (h/use-did-update (fn [] (println "Printed on component-did-update")))
```

Adds a callback for the componentDidUpdate lifecycle hook.


### The 'Higher-Level' API

Described below are two higher level hooks created out of the low level hooks.

### use-effect

```clojure
  (let [_ (h/use-effect (fn [] (println "Printed on component-did-mount"))
                          (fn [] (println "Printed on component-will-umount") ))])
```

a wrapper around use-did-mount and use-will-unmount...

given a function it will call that function on did-mount.

### use-watch

```clojure
  (let [some-atom (r/atom 42)
        _ (h/use-watch some-atom (fn [new-value]
                                   (println "Value changed to" new-value)))])
```

A wrapper around atoms and use-effect.
adds a watcher to the given atom. executes the callback new-value of atom. removes watcher from atom on component unmount.

## Creating your own hooks (plus some example usages for other hooks)

The whole point of hooks is to be extensible. Hooks can (and should) be use to create more specific hooks.

for example:


Say we had a hook that could open a core.async channel on componentDidMount and close it componentWillUnmount.

```clojure
(defn use-chan 

  "Returns a channel that closes when component is removed from dom. 
  Takes the same arguments a clojure.core.async/chan function"

  [& args]
  (let [channel (apply async/chan args)]
    (h/use-will-unmount #(async/close! channel))
    channel))

```

Once we have this function (feel free to use it btw :P)

We can make a higher-level (well a more higher-level, as use-chan is it's self higher-level) hook called use-pipeline.
(again feel free to use this one as well, not like i could stop you anyway :P)

```clojure
(defn use-pipeline [paralelism async-fn!]
  (let [input (use-chan paralelism)
        output (use-chan paralelism)]

    (async/pipeline-async paralelism output async-fn! input)

    [input output]))

```

After this we can make an even higher higher-level hook that puts it all together... (you can use this one as well, but i think you're better off making your own in this particular case)

```clojure
(defn use-livesearch [search-fn!]
  (let [[text set-text!] (h/use-state "")
        [results set-results!] (h/use-state [])
        [input output] (use-pipeline 4 search-fn!)]

    (h/use-watch text (fn [new-text] (async/put! input new-text)))

    (go-loop [] ;; async loop will terminated on will-unmount because channels will close
      (when-let [r (<! output)]
         (set-results! r)
         (recur)))

    [text #(-> % .-target .-value set-text!) results])) ;; returns the text ratom, a setter that knows about text-inputs and the list of results

```

Basically we now have a `simple` (god i hope i'm using this term right :P) way to do something like live search like so:

```clojure
(defn live-search-fn! [input result-chan]
   ;;; do actual search and when result comes in put it on the result-chan and close it
   ;;; see clojure.core.async/pipeline-async for more details 
) 

(defn some-search-box-with-results []
  (let [[text set-text! results] (use-livesearch live-search-fn!)]
    (fn [] 
      [:div
        [:input {:value @text :on-change set-text!}]
        [:div @results]] )

```



## Setup

```clojure
  [boogie666/reagent-hooks "0.1.0"]
```

## License (same as clojure)

Copyright © 2019 boogie666

Distributed under the Eclipse Public License either version 1.0 or (at your option) any later version.
