(ns blueprints.clj
  (:import (com.tinkerpop.blueprints Graph MetaGraph
                                     IndexableGraph KeyIndexableGraph
                                     TransactionalGraph ThreadedTransactionalGraph
                                     Element Vertex Edge Direction
                                     Query Query$Compare VertexQuery
                                     Features)
           (com.tinkerpop.blueprints.util GraphHelper)
           (com.tinkerpop.blueprints.util.wrappers WrapperGraph)
           (com.tinkerpop.blueprints.util.wrappers.batch BatchGraph
                                                         VertexIDType)
           (com.tinkerpop.blueprints.util.wrappers.event EventGraph
                                                         EventTransactionalGraph
                                                         EventTransactionalIndexableGraph)
           com.tinkerpop.blueprints.util.wrappers.partition.PartitionGraph
           (com.tinkerpop.blueprints.util.wrappers.id IdGraph
                                                      IdGraph$IdFactory)
           (com.tinkerpop.blueprints.util.wrappers.readonly ReadOnlyGraph
                                                            ReadOnlyIndexableGraph
                                                            ReadOnlyKeyIndexableGraph)
           com.tinkerpop.blueprints.util.wrappers.event.listener.GraphChangedListener)
  (:use clojure.template)
  (:require [clojure.string :as string]))

(declare vertex? edge?
         ->java  ->clj
         wrap    unwrap)

;; [State]
(def ^:dynamic ^Graph ^{:doc "The database connection in current use."} *db*)

;; [Utils]
(def ^:private keyword->direction
  {:in   Direction/IN
   :out  Direction/OUT
   :both Direction/BOTH})

(defn ^:private ^Query query<== [^Query query hmap]
  (doseq [[k* v] hmap]
    (case k*
      :$limit     (.limit query v)
      :$direction (.direction ^VertexQuery query (keyword->direction v))
      :$labels    (.labels ^VertexQuery query (into-array v))
      (if (vector? v)
        (let [k (name k*)
              [op v1 v2] v]
          (case op
            :$=        (.has query k v1 Query$Compare/EQUAL)
            :$not=     (.has query k v1 Query$Compare/NOT_EQUAL)
            :$<        (.has query k v1 Query$Compare/LESS_THAN)
            :$<=       (.has query k v1 Query$Compare/LESS_THAN_EQUAL)
            :$>        (.has query k v1 Query$Compare/GREATER_THAN)
            :$>=       (.has query k v1 Query$Compare/GREATER_THAN_EQUAL)
            :$interval (.interval query k v1 v2)))
        (.has query (name k*) v))))
  query)

(defn ^:private =query<== [query hmap]
  `(doto ~query
     ~@(mapcat identity
               (for [[k* v] hmap]
                 (case k*
                   :$limit     `(.limit ~v)
                   :$direction `(.direction ~(keyword->direction v))
                   :$labels    `(.labels ~(into-array v))
                   (if (vector? v)
                     (let [k (name k*)
                           [op v1 v2] v]
                       (case op
                         :$=        `(.has ~k ~v1 Query$Compare/EQUAL)
                         :$not=     `(.has ~k ~v1 Query$Compare/NOT_EQUAL)
                         :$<        `(.has ~k ~v1 Query$Compare/LESS_THAN)
                         :$<=       `(.has ~k ~v1 Query$Compare/LESS_THAN_EQUAL)
                         :$>        `(.has ~k ~v1 Query$Compare/GREATER_THAN)
                         :$>=       `(.has ~k ~v1 Query$Compare/GREATER_THAN_EQUAL)
                         :$interval `(.interval ~k ~v1 ~v2)))
                     `(.has ~(name k*) ~v)))))))

(defn ^:private id-factory
  "ID factories for as-id"
  [f]
  (proxy [IdGraph$IdFactory] []
         (createId [] (f))))

(def ^:private graph-event->method
  '{:vertex/add              [vertexAdded (VertexWrapper. vertex)]
    :vertex/removed          [vertexRemoved (VertexWrapper. vertex)]
    :vertex/property-changed [vertexPropertyChanged (VertexWrapper. vertex) (keyword key) old-val set-val]
    :vertex/property-removed [vertexPropertyRemoved (VertexWrapper. vertex) (keyword key) removed-val]
    :edge/add                [edgeAdded (EdgeWrapper. edge)]
    :edge/removed            [edgeRemoved (EdgeWrapper. edge)]
    :edge/property-changed   [edgePropertyChanged (EdgeWrapper. edge) (keyword key) old-val set-val]
    :edge/property-removed   [edgePropertyRemoved (EdgeWrapper. edge) (keyword key) removed-val]})

(defn ^:private handlers->graph-changed-listener [handlers]
  {:pre [(every? #=(set (keys graph-event->method)) (keys handlers))]}
  (let [pseudonyms (map #(gensym (str (namespace %) "-" (name %)))
                        (keys handlers))
        handlers* (into {}
                        (map vector
                             (keys handlers)
                             pseudonyms))]
    `(let [~@(mapcat vector
                     pseudonyms
                     (vals handlers))]
       (reify GraphChangedListener
         ~@(for [[event handler] handlers*
                 :let [[method & args] (graph-event->method event)]]
             `(~method [self# ~@(map #(if (seq? %) (second %) %) args)]
                       (~handler ~@args))))
       )))

;; Wrappers
(defprotocol ElementWrapper)

(do-template [<sym> <type-pred> <case>]
  (deftype <sym> [obj]
    ElementWrapper
    clojure.lang.IPersistentMap
    (assoc [self k v]
      (.setProperty ^Element obj (name k) v)
      self)
    (without [self k]
      (.removeProperty ^Element obj (name k))
      self)
    
    java.lang.Iterable
    (iterator [self]
      (.iterator ^Iterable (.seq self)))
    
    clojure.lang.Associative
    (containsKey [self k]
      (boolean (.getProperty ^Element obj (name k))))
    (entryAt [self k]
      (if-let [v (.valAt self k)]
        (clojure.lang.MapEntry. k v)))
    
    clojure.lang.IPersistentCollection
    (count [self]
      (count (.getPropertyKeys ^Element obj)))
    (cons [self o]
      (doseq [[k v] o]
        (.assoc self k v))
      self)
    (equiv [self o]
      (.equals self o))
    
    clojure.lang.Seqable
    (seq [self]
      (for [k (.getPropertyKeys ^Element obj)
            :let [k (keyword k)]]
        (clojure.lang.MapEntry. k (.valAt self k))))
    
    clojure.lang.ILookup
    (valAt [self k not-found]
      (or (.valAt self k)
          not-found))
    (valAt [self k]
      <case>)
    
    java.lang.Object
    (equals [self o]
      (and (<type-pred> o)
           (.equals obj (.obj ^EdgeWrapper o)))))

  EdgeWrapper edge?
  (case k
    :&id    (.getId ^Edge obj)
    :&label (.getLabel ^Edge obj)
    :&in    (wrap (.getVertex ^Edge obj Direction/IN))
    :&out   (wrap (.getVertex ^Edge obj Direction/OUT))
    (.getProperty ^Edge obj (name k)))

  VertexWrapper vertex?
  (case k
    :&id (.getId ^Vertex obj)
    :&in    (map wrap (.getVertices ^Vertex obj Direction/IN  (into-array [])))
    :&out   (map wrap (.getVertices ^Vertex obj Direction/OUT (into-array [])))
    (.getProperty ^Vertex obj (name k)))
  )

(defn ^:private ->java [x]
  (cond (instance? ElementWrapper x) (unwrap x)
        (map? x)                     (->> x
                                          (map (fn [[k v]]
                                                 [(name k) (->java v)]))
                                          (into {}))
        (coll? x)                    (map ->java x)
        :else                        x))

(defn ^:private ->clj [x]
  (cond (instance? Element x)        (wrap x)
        (instance? java.util.Map x)  (reduce (fn [total k]
                                               (assoc total (keyword k) (->clj (.get ^java.util.Map x k))))
                                             {}
                                             (-> ^java.util.Map x
                                                 .keySet
                                                 .iterator
                                                 iterator-seq))
        (instance? java.util.List x) (map ->clj x)
        :else                        x))

;; [Interface]
;; Wrappers
(defn ^blueprints.clj.ElementWrapper wrap
  "Wraps a Blueprints vertex|edge in a custom data-type that behaves like a Clojure map."
  [^Element elem]
  (condp instance? elem
    Vertex (VertexWrapper. elem)
    Edge   (EdgeWrapper. elem)))

(defn ^Element unwrap
  "Returns the original Blueprints vertex|dge object that has been wrapped."
  [^blueprints.clj.ElementWrapper elem]
  (.-obj elem))

;; Syntax
(defmacro with-db
  "Evaluates body with a globally available DB connection."
  [db & body]
  `(binding [*db* ~db]
     (try ~@body
       (finally (.shutdown *db*)))))

(defmacro tx
  "Commits the succesful transaction when done and does a rollback if an exception recur, re-throwing it for later catching."
  [& body]
  `(try (let [res# (do ~@body)]
          (.commit ^TransactionalGraph *db*)
          res#)
     (catch Exception e#
       (.rollback ^TransactionalGraph *db*)
       (throw e#))))

;; Fns
(defmacro ^:private __features=> [features]
  (let [g!features (gensym "features")]
    `(defn ~(with-meta 'features=> {:doc "Returns a map with the supported features of the DB as K/V pairs."})
       []
       (let [~g!features ^Features (.getFeatures *db*)]
         ~(->> (for [feat features]
                 [(-> feat
                      str
                      (string/replace #"((?=[A-Z]))" "$1-")
                      string/lower-case
                      keyword)
                  `(. ~g!features ~feat)])
               (into {}))))))

(__features=> [isPersistent isWrapper ignoresSuppliedIds supportsSelfLoops
               supportsStringProperty supportsSerializableObjectProperty
               supportsBooleanProperty supportsIntegerProperty supportsLongProperty supportsFloatProperty supportsDoubleProperty
               supportsPrimitiveArrayProperty supportsUniformListProperty supportsMixedListProperty supportsMapProperty
               supportsEdgeRetrieval supportsDuplicateEdges
               supportsIndices supportsKeyIndices supportsVertexIndex supportsVertexKeyIndex supportsEdgeIndex supportsEdgeKeyIndex
               supportsVertexProperties supportsVertexIteration supportsEdgeProperties supportsEdgeIteration
               supportsTransactions supportsThreadedTransactions])

(defn raw-graph=> "Returns the raw graph of a MetaGraph."
  []
  (.getRawGraph ^MetaGraph *db*))

(defn base-graph=> "Returns the base graph of a WrapperGraph."
  []
  (.getBaseGraph ^WrapperGraph *db*))

(defn vertex
  "Instantiates a vertex in the DB. To provide an ID, pass it with the attributes under the :&id key."
  ([]      (vertex {}))
  ([attrs] (merge (VertexWrapper. (.addVertex *db* (attrs :&id)))
                  (dissoc attrs :&id))))

(defn edge
  "Instantiates an edge in the DB. To provide an ID, pass it with the attributes under the :&id key."
  ([^VertexWrapper v1 ^VertexWrapper v2 ^String label]
     (edge v1 v2 label nil))
  ([^VertexWrapper v1 ^VertexWrapper v2 ^String label attrs]
     (merge (EdgeWrapper. (.addEdge *db* (:&id attrs) ^Vertex (.obj v1) ^Vertex (.obj v2) label))
            (dissoc attrs :&id))))

(do-template [<sym> <type>]
  (defn <sym> "" [x]
    (instance? <type> x))
  
  vertex? VertexWrapper
  edge?   EdgeWrapper)

(defn threaded-tx
  "Returns a new treaded transactional graph that can be bound using with-db."
  []
  (.newTransaction ^ThreadedTransactionalGraph *db*))

(defn edges=>
  "Returns the edges to|from a vertex."
  [^VertexWrapper vertex direction labels]
  (->> (.getEdges ^Vertex (.obj vertex) (keyword->direction direction)
                  (into-array labels))
       .iterator
       iterator-seq
       (map wrap)))

(defn vertices=> "Returns the vertices at the ends of the edges going to|from a vertex."
  [^VertexWrapper vertex direction labels]
  (->> (.getVertices ^Vertex (.obj vertex) (keyword->direction direction)
                     (into-array labels))
       .iterator
       iterator-seq
       (map wrap)))

(do-template [<sym> <method> <doc>]
  (defn <sym> <doc> [id]
    (wrap (<method> *db* id)))

  edge=>    .getEdge   "Returns edge with the given ID from the DB."
  vertex=>  .getVertex "Returns vertex with the given ID from the DB.")

(defn remove! "" [^blueprints.clj.ElementWrapper element]
  (.remove ^Element (.obj element)))

(defn copy! "" [^Graph from ^Graph to]
  (GraphHelper/copyGraph from to))

(defn query "Performs a DB query, where type is one of #{:vertex, :edge}, query is a map and source can be a graph element."
  ([type query] (query *db* type query))
  ([source type query]
     (let [query* (query<== (.query source) query)]
       (->> (case type
              :vertex (.vertices query*)
              :edge   (.edges query*))
            .iterator
            iterator-seq
            (map wrap)))))

(defmacro =query "Same as 'query', but transforms query-map into a Query object at compile-time."
  ([type query] '(=query *db* ~type ~query))
  ([source type query]
     `(->> ~(case type
              :vertex `(.vertices ~(=query<== `(.query ~source) query))
              :edge   `(.edges ~(=query<== `(.query ~source) query)))
           .iterator
           iterator-seq
           (map wrap))))

;; Graph DB behavioral wrappers.
(defmacro as-batch "Evaluates body while treating the DB as a BatchGraph."
  [args & body]
  `(let [[^VertexIDType vertex-id-type# ^Long buffer-size#] ~args]
     (binding [*db* (BatchGraph. *db* (or vertex-id-type#
                                          VertexIDType/OBJECT)
                                 (or buffer-size#
                                     BatchGraph/DEFAULT_BUFFER_SIZE))]
       ~@body)))

(defmacro as-id "Evaluates body while treating the DB as a IdGraph."
  [args & body]
  `(let [{vertex-id-factory# :vertex-id-factory edge-id-factory# :edge-id-factory} ~args]
     (binding [*db* (IdGraph. *db* (boolean vertex-id-factory#) (boolean edge-id-factory#))]
       (when (fn? vertex-id-factory#) (.setVertexIdFactory (id-factory vertex-id-factory#)))
       (when (fn? edge-id-factory#)   (.setEdgeIdFactory (id-factory edge-id-factory#)))
       ~@body)))

(defmacro as-partition "Evaluates body while treating the DB as a PartitionGraph."
  [args & body]
  `(let [[^String key# ^String write# ^java.collection.Set read#] ~args]
     (binding [*db* (PartitionGraph. *db* key# write# read#)]
       ~@body)))

(defmacro as-read-only "Evaluates body while treating the DB as a ReadOnlyGraph."
  [& body]
  `(binding [*db* (cond (instance? IndexableGraph *db*)    (ReadOnlyIndexableGraph. *db*)
                        (instance? KeyIndexableGraph *db*) (ReadOnlyKeyIndexableGraph. *db*)
                        :else                              (ReadOnlyGraph. *db*))]
     ~@body))

(defmacro as-evented "Evaluates body while treating the DB as a EventGraph or one of its variants (EventTransactionalIndexableGraph or EventTransactionalGraph)."
  [handlers & body]
  `(binding [*db* (if (instance? TransactionalGraph *db*)
                    (if (instance? IndexableGraph *db*)
                      (EventTransactionalIndexableGraph. *db*)
                      (EventTransactionalGraph. *db*))
                    (EventGraph. *db*))]
     (.addListener ^EventGraph *db* ~(handlers->graph-changed-listener handlers))
     ~@body))

;; [Omissions]
(def ^:no-doc ->EdgeWrapper)
(def ^:no-doc ->VertexWrapper)
