(ns chromex.app.platform-keys-internal
  "Internal API for to implement the platformKeys and enterprise.platformKeys
   APIs.
   
     * available since Chrome 45
     * https://developer.chrome.com/extensions/platformKeysInternal"

  (:refer-clojure :only [defmacro defn apply declare meta let])
  (:require [chromex.wrapgen :refer [gen-wrap-from-table]]
            [chromex.callgen :refer [gen-call-from-table gen-tap-all-call]]
            [chromex.config :refer [get-static-config gen-active-config]]))

(declare api-table)
(declare gen-call)

; -- functions --------------------------------------------------------------------------------------------------------------

(defmacro select-client-certificates
  "See documentation in platformKeys.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([details #_callback] (gen-call :function ::select-client-certificates &form details)))

(defmacro sign
  "Internal version of platformKeys.subtleCrypto.sign and enterprise.platformKeys.Token.subtleCrypto.sign. |tokenId| The id of
   a Token returned by |getTokens|. |publicKey| The Subject Public Key Info of a key previously generated by     |generateKey

   in DER encoding. |hashAlgorithmName| The recognized algorithm name of the hash algorithm,     as specified by WebCrypto,
   that will be used to digest |data|     before signing. Currently supported are: SHA-{1,256,384,512}.     If instead the
   algorithm name 'none' is provided, no hashing will be     applied, the data is PKCS#1 v1.5 padded but not hashed.
   TODO(pneubeck): use an enum once supported:         http://www.crbug.com/385539 . |data| The data to sign. |callback

   Called back with the signature of |data|. TODO: Instead of ArrayBuffer should be (ArrayBuffer or ArrayBufferView), or at
   least (ArrayBuffer or Uint8Array).
   
     |callback| - Invoked by sign. |signature| The signature, a octet string.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([token-id public-key hash-algorithm-name data #_callback] (gen-call :function ::sign &form token-id public-key hash-algorithm-name data)))

(defmacro get-public-key
  "Checks whether certificate certifies a key that allows usage of the WebCrypto algorithm algorithmName. If so, calls back
   callback with the key info and a WebCrypto KeyAlgorithm dictionary describing the key's algorithm. The name property will
   equal algorithmName. Otherwise, calls back with an error.
   
     |callback| - Called back by getPublicKey. |publicKey| The Subject Public Key Info (see X.509) of the requested
                  certificate. |algorithm| A partial WebCrypto KeyAlgorithm containing all information   that is available
                  from the Subject Public Key Info. It does not contain   signature/hash parameters.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([certificate algorithm-name #_callback] (gen-call :function ::get-public-key &form certificate algorithm-name)))

; -- convenience ------------------------------------------------------------------------------------------------------------

(defmacro tap-all-events
  "Taps all valid non-deprecated events in this namespace."
  [chan]
  (let [static-config (get-static-config)
        config (gen-active-config static-config)]
    (gen-tap-all-call static-config api-table (meta &form) config chan)))

; ---------------------------------------------------------------------------------------------------------------------------
; -- API TABLE --------------------------------------------------------------------------------------------------------------
; ---------------------------------------------------------------------------------------------------------------------------

(def api-table
  {:namespace "chrome.platformKeysInternal",
   :since "45",
   :functions
   [{:id ::select-client-certificates,
     :name "selectClientCertificates",
     :callback? true,
     :params
     [{:name "details", :type "platformKeys.SelectDetails"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "certs", :type "[array-of-platformKeys.Matchs]"}]}}]}
    {:id ::sign,
     :name "sign",
     :callback? true,
     :params
     [{:name "token-id", :type "string"}
      {:name "public-key", :type "ArrayBuffer"}
      {:name "hash-algorithm-name", :type "string"}
      {:name "data", :type "ArrayBuffer"}
      {:name "callback", :type :callback, :callback {:params [{:name "signature", :type "ArrayBuffer"}]}}]}
    {:id ::get-public-key,
     :name "getPublicKey",
     :callback? true,
     :params
     [{:name "certificate", :type "ArrayBuffer"}
      {:name "algorithm-name", :type "string"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "public-key", :type "ArrayBuffer"} {:name "algorithm", :type "object"}]}}]}]})

; -- helpers ----------------------------------------------------------------------------------------------------------------

; code generation for native API wrapper
(defmacro gen-wrap [kind item-id config & args]
  (let [static-config (get-static-config)]
    (apply gen-wrap-from-table static-config api-table kind item-id config args)))

; code generation for API call-site
(defn gen-call [kind item src-info & args]
  (let [static-config (get-static-config)
        config (gen-active-config static-config)]
    (apply gen-call-from-table static-config api-table kind item src-info config args)))