(ns chromex.app.easy-unlock-private
  "chrome.easyUnlockPrivate API that provides hooks to Chrome to
   be used by Easy Unlock component app.
   
     * available since Chrome 38
     * https://developer.chrome.com/extensions/easyUnlockPrivate"

  (:refer-clojure :only [defmacro defn apply declare meta let])
  (:require [chromex.wrapgen :refer [gen-wrap-from-table]]
            [chromex.callgen :refer [gen-call-from-table gen-tap-all-call]]
            [chromex.config :refer [get-static-config gen-active-config]]))

(declare api-table)
(declare gen-call)

; -- functions --------------------------------------------------------------------------------------------------------------

(defmacro get-strings
  "Gets localized strings required to render the API.
   
     |callback| - Called with a dictionary mapping names to resource strings. TODO(isherman): This is essentially copied
                  from identity_private.idl.                 Perhaps this should be extracted to a common API instead?
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([#_callback] (gen-call :function ::get-strings &form)))

(defmacro generate-ec-p256-key-pair
  "Generates a ECDSA key pair for P256 curve. Public key will be in format recognized by secure wire transport protocol used
   by Easy Unlock app. Otherwise, the exact format for both key should should be considered obfuscated to the app. The app
   should not use them directly, but through this API.
   
     |callback| - Callback with the generated keys. On failure, none of the     keys will be set.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([#_callback] (gen-call :function ::generate-ec-p256-key-pair &form)))

(defmacro perform-ecdh-key-agreement
  "Given a private key and a public ECDSA key from different asymetric key pairs, it generates a symetric encryption key using
   EC Diffie-Hellman scheme.
   
     |privateKey| - A private key generated by the app using     |generateEcP256KeyPair|.
     |publicKey| - A public key that should be in the same format as the     public key generated by
                   |generateEcP256KeyPair|. Generally not the     one paired with |private_key|.
     |callback| - Function returning the generated secret symetric key.     On failure, the returned value will not be set.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([private-key public-key #_callback] (gen-call :function ::perform-ecdh-key-agreement &form private-key public-key)))

(defmacro create-secure-message
  "Creates a secure, signed message in format used by Easy Unlock app to establish secure communication channel over unsecure
   connection.
   
     |payload| - The payload the create message should carry.
     |key| - The key used to sign the message content. If encryption algorithm     is set in |options| the same key will be
             used to encrypt the message.
     |options| - Additional (optional) parameters used to create the message.
     |callback| - Function returning the created message bytes. On failure,     the returned value will not be set.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([payload key options #_callback] (gen-call :function ::create-secure-message &form payload key options)))

(defmacro unwrap-secure-message
  "Authenticates and, if needed, decrypts a secure message. The message is in the same format as the one created by
   |createSecureMessage|.
   
     |secureMessage| - The message to be unwrapped.
     |key| - Key to be used to authenticate the message sender. If encryption     algorithm is set in |options|, the same
             key will be used to decrypt     the message.
     |options| - Additional (optional) parameters used to unwrap the message.
     |callback| - Function returning an array buffer containing cleartext     message header and body. They are returned in
                  a single buffer in     format used inside the message. If the massage authentication or     decryption
                  fails, the returned value will not be set.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([secure-message key options #_callback] (gen-call :function ::unwrap-secure-message &form secure-message key options)))

(defmacro seek-bluetooth-device-by-address
  "Connects to the SDP service on a device, given just the device's Bluetooth address. This function is useful as a faster
   alternative to Bluetooth discovery, when you already know the remote device's Bluetooth address. A successful call to this
   function has the side-effect of registering the device with the Bluetooth daemon, making it available for future outgoing
   connections.
   
     |deviceAddress| - The Bluetooth address of the device to connect to.
     |callback| - Called to indicate success or failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([device-address #_callback] (gen-call :function ::seek-bluetooth-device-by-address &form device-address)))

(defmacro connect-to-bluetooth-service-insecurely
  "Connects the socket to a remote Bluetooth device over an insecure connection, i.e. a connection that requests no bonding
   and no man-in-the-middle protection. Other than the reduced security setting, behaves identically to the
   chrome.bluetoothSocket.connect() function.
   
     |socketId| - The socket identifier, as issued by the     chrome.bluetoothSocket API.
     |deviceAddress| - The Bluetooth address of the device to connect to.
     |uuid| - The UUID of the service to connect to.
     |callback| - Called when the connect attempt is complete.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([socket-id device-address uuid #_callback] (gen-call :function ::connect-to-bluetooth-service-insecurely &form socket-id device-address uuid)))

(defmacro update-screenlock-state
  "Updates the screenlock state to reflect the Easy Unlock app state.
   
     |callback| - An empty callback used purely for signalling success vs. failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([state #_callback] (gen-call :function ::update-screenlock-state &form state)))

(defmacro set-permit-access
  "Saves the permit record for the local device.
   
     |permitAccess| - The permit record to be saved.
     |callback| - Called to indicate success or failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([permit-access #_callback] (gen-call :function ::set-permit-access &form permit-access)))

(defmacro get-permit-access
  "Gets the permit record for the local device.
   
     |callback| - Callback for the getPermitAccess() method.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([#_callback] (gen-call :function ::get-permit-access &form)))

(defmacro clear-permit-access
  "Clears the permit record for the local device.
   
     |callback| - An empty callback used purely for signalling success vs. failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([#_callback] (gen-call :function ::clear-permit-access &form)))

(defmacro set-remote-devices
  "Saves the remote device list.
   
     |devices| - The list of remote devices to be saved.
     |callback| - Called to indicate success or failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([devices #_callback] (gen-call :function ::set-remote-devices &form devices)))

(defmacro get-remote-devices
  "Gets the remote device list.
   
     |callback| - Callback for the getRemoteDevices() method.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([#_callback] (gen-call :function ::get-remote-devices &form)))

(defmacro get-sign-in-challenge
  "Gets the sign-in challenge for the current user.
   
     |nonce| - Nonce that should be signed by the Chrome OS TPM. The signed     nonce is returned with the sign-in challenge.
     |callback| - Callback for |getSignInChallenge()| method. In case challenge could not be created both |challenge| and
                  |signedNonce| will be empty. If the requested nonce could not be signed, |signedNonce| will be empty.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([nonce #_callback] (gen-call :function ::get-sign-in-challenge &form nonce)))

(defmacro try-sign-in-secret
  "Tries to sign-in the current user with a secret obtained by decrypting the sign-in challenge. Check
   chrome.runtime.lastError for failures. Upon success, the user session will be started.
   
     |callback| - An empty callback used purely for signalling success vs. failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([sign-in-secret #_callback] (gen-call :function ::try-sign-in-secret &form sign-in-secret)))

(defmacro get-user-info
  "Retrieves information about the user associated with the Easy unlock service.
   
     |callback| - Callback for the |getUserInfo()| method. Note that the callback argument is a list for future use (on
                  signin screen there may be more than one user associated with the easy unlock service). Currently the
                  method returns at most one user.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([#_callback] (gen-call :function ::get-user-info &form)))

(defmacro get-connection-info
  "Gets the connection info for the Bluetooth device identified by deviceAddress.
   
     |callback| - Callback for the |getConnectionInfo()| method.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([device-address #_callback] (gen-call :function ::get-connection-info &form device-address)))

(defmacro show-error-bubble
  "Shows an error bubble with the given |message|, anchored to an edge of the given |anchorRect| -- typically the right edge,
   but possibly a different edge if there is not space for the bubble to the right of the anchor rectangle. If the
   |link_range| is non-empty, renders the text within the |message| that is contained in the |link_range| as a link with the
   given |link_target| URL."
  ([message link-range link-target anchor-rect] (gen-call :function ::show-error-bubble &form message link-range link-target anchor-rect)))

(defmacro hide-error-bubble
  "Hides the currently visible error bubble, if there is one."
  ([] (gen-call :function ::hide-error-bubble &form)))

(defmacro set-auto-pairing-result
  "Sets the result of auto pairing triggered from onStartAutoPairing event. If auto pairing is completed successfully,
   |result.success| should be true so that Easy bootstrap flow would finish and starts the user session. Otherwise,
   |result.success| is set to false with an optional error message to be displayed to the user.
   
     |callback| - An empty callback used purely for signalling success vs. failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([result #_callback] (gen-call :function ::set-auto-pairing-result &form result)))

(defmacro find-setup-connection
  "Finds and connects the remote BLE device that is advertising: |setupServiceUUID|. Returns when a connection is found or
   |timeOut| seconds have elapsed.
   
     |callback| - Callback for the |FindSetupConnectionCallback| method.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([setup-service-uuid time-out #_callback] (gen-call :function ::find-setup-connection &form setup-service-uuid time-out)))

(defmacro setup-connection-status
  "Returns the status of the connection with |connectionId|.
   
     |callback| - Callback for the |setupConnectionStatus()| method.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([connection-id #_callback] (gen-call :function ::setup-connection-status &form connection-id)))

(defmacro setup-connection-disconnect
  "Disconnects the connection with |connectionId|.
   
     |callback| - An empty callback used purely for signalling success vs. failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([connection-id #_callback] (gen-call :function ::setup-connection-disconnect &form connection-id)))

(defmacro setup-connection-send
  "Sends |data| through the connection with |connnectionId|.
   
     |callback| - An empty callback used purely for signalling success vs. failure.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([connection-id data #_callback] (gen-call :function ::setup-connection-send &form connection-id data)))

(defmacro setup-connection-get-device-address
  "Gets the Bluetooth address of the connection with |connectionId

   
     |callback| - Callback for the |setupConnectionGetDeviceAddress()| method.
   
   Note: Instead of passing a callback function, you receive a core.async channel as return value."
  ([connection-id #_callback] (gen-call :function ::setup-connection-get-device-address &form connection-id)))

; -- events -----------------------------------------------------------------------------------------------------------------
;
; docs: https://github.com/binaryage/chromex/#tapping-events

(defmacro tap-on-user-info-updated-events
  "Event fired when the data for the user currently associated with Easy unlock service is updated. |userInfo| The updated
   user information.
   Events will be put on the |channel|.
   
   Note: |args| will be passed as additional parameters into Chrome event's .addListener call."
  ([channel & args] (apply gen-call :event ::on-user-info-updated &form channel args)))
(defmacro tap-on-start-auto-pairing-events
  "Event fired at the end of Easy bootstrap to start auto pairing so that a proper cryptohome key could be generated for the
   user.
   Events will be put on the |channel|.
   
   Note: |args| will be passed as additional parameters into Chrome event's .addListener call."
  ([channel & args] (apply gen-call :event ::on-start-auto-pairing &form channel args)))
(defmacro tap-on-connection-status-changed-events
  "Event fired when |connectionId| change status.
   Events will be put on the |channel|.
   
   Note: |args| will be passed as additional parameters into Chrome event's .addListener call."
  ([channel & args] (apply gen-call :event ::on-connection-status-changed &form channel args)))
(defmacro tap-on-data-received-events
  "Event fired when |connectionId| receives |data|.
   Events will be put on the |channel|.
   
   Note: |args| will be passed as additional parameters into Chrome event's .addListener call."
  ([channel & args] (apply gen-call :event ::on-data-received &form channel args)))
(defmacro tap-on-send-completed-events
  "Event fired when |connectionId| sends |data|. |success| is true if the send operation was successful.
   Events will be put on the |channel|.
   
   Note: |args| will be passed as additional parameters into Chrome event's .addListener call."
  ([channel & args] (apply gen-call :event ::on-send-completed &form channel args)))

; -- convenience ------------------------------------------------------------------------------------------------------------

(defmacro tap-all-events
  "Taps all valid non-deprecated events in this namespace."
  [chan]
  (let [static-config (get-static-config)
        config (gen-active-config static-config)]
    (gen-tap-all-call static-config api-table (meta &form) config chan)))

; ---------------------------------------------------------------------------------------------------------------------------
; -- API TABLE --------------------------------------------------------------------------------------------------------------
; ---------------------------------------------------------------------------------------------------------------------------

(def api-table
  {:namespace "chrome.easyUnlockPrivate",
   :since "38",
   :functions
   [{:id ::get-strings,
     :name "getStrings",
     :callback? true,
     :params [{:name "callback", :type :callback, :callback {:params [{:name "strings", :type "object"}]}}]}
    {:id ::generate-ec-p256-key-pair,
     :name "generateEcP256KeyPair",
     :callback? true,
     :params
     [{:name "callback",
       :type :callback,
       :callback
       {:params
        [{:name "public-key", :optional? true, :type "ArrayBuffer"}
         {:name "private-key", :optional? true, :type "ArrayBuffer"}]}}]}
    {:id ::perform-ecdh-key-agreement,
     :name "performECDHKeyAgreement",
     :callback? true,
     :params
     [{:name "private-key", :type "ArrayBuffer"}
      {:name "public-key", :type "ArrayBuffer"}
      {:name "callback", :type :callback, :callback {:params [{:name "data", :optional? true, :type "ArrayBuffer"}]}}]}
    {:id ::create-secure-message,
     :name "createSecureMessage",
     :callback? true,
     :params
     [{:name "payload", :type "ArrayBuffer"}
      {:name "key", :type "ArrayBuffer"}
      {:name "options", :type "object"}
      {:name "callback", :type :callback, :callback {:params [{:name "data", :optional? true, :type "ArrayBuffer"}]}}]}
    {:id ::unwrap-secure-message,
     :name "unwrapSecureMessage",
     :callback? true,
     :params
     [{:name "secure-message", :type "ArrayBuffer"}
      {:name "key", :type "ArrayBuffer"}
      {:name "options", :type "object"}
      {:name "callback", :type :callback, :callback {:params [{:name "data", :optional? true, :type "ArrayBuffer"}]}}]}
    {:id ::seek-bluetooth-device-by-address,
     :name "seekBluetoothDeviceByAddress",
     :callback? true,
     :params [{:name "device-address", :type "string"} {:name "callback", :optional? true, :type :callback}]}
    {:id ::connect-to-bluetooth-service-insecurely,
     :name "connectToBluetoothServiceInsecurely",
     :callback? true,
     :params
     [{:name "socket-id", :type "integer"}
      {:name "device-address", :type "string"}
      {:name "uuid", :type "string"}
      {:name "callback", :type :callback}]}
    {:id ::update-screenlock-state,
     :name "updateScreenlockState",
     :callback? true,
     :params [{:name "state", :type "unknown-type"} {:name "callback", :optional? true, :type :callback}]}
    {:id ::set-permit-access,
     :name "setPermitAccess",
     :callback? true,
     :params
     [{:name "permit-access", :type "easyUnlockPrivate.PermitRecord"}
      {:name "callback", :optional? true, :type :callback}]}
    {:id ::get-permit-access,
     :name "getPermitAccess",
     :callback? true,
     :params
     [{:name "callback",
       :type :callback,
       :callback {:params [{:name "permit-access", :optional? true, :type "easyUnlockPrivate.PermitRecord"}]}}]}
    {:id ::clear-permit-access,
     :name "clearPermitAccess",
     :callback? true,
     :params [{:name "callback", :optional? true, :type :callback}]}
    {:id ::set-remote-devices,
     :name "setRemoteDevices",
     :callback? true,
     :params
     [{:name "devices", :type "[array-of-easyUnlockPrivate.Devices]"}
      {:name "callback", :optional? true, :type :callback}]}
    {:id ::get-remote-devices,
     :name "getRemoteDevices",
     :callback? true,
     :params
     [{:name "callback",
       :type :callback,
       :callback {:params [{:name "devices", :type "[array-of-easyUnlockPrivate.Devices]"}]}}]}
    {:id ::get-sign-in-challenge,
     :name "getSignInChallenge",
     :callback? true,
     :params
     [{:name "nonce", :type "ArrayBuffer"}
      {:name "callback",
       :type :callback,
       :callback
       {:params
        [{:name "challenge", :optional? true, :type "ArrayBuffer"}
         {:name "signed-nonce", :optional? true, :type "ArrayBuffer"}]}}]}
    {:id ::try-sign-in-secret,
     :name "trySignInSecret",
     :callback? true,
     :params [{:name "sign-in-secret", :type "ArrayBuffer"} {:name "callback", :type :callback}]}
    {:id ::get-user-info,
     :name "getUserInfo",
     :callback? true,
     :params
     [{:name "callback",
       :type :callback,
       :callback {:params [{:name "users", :type "[array-of-easyUnlockPrivate.UserInfos]"}]}}]}
    {:id ::get-connection-info,
     :name "getConnectionInfo",
     :callback? true,
     :params
     [{:name "device-address", :type "string"}
      {:name "callback",
       :type :callback,
       :callback
       {:params
        [{:name "rssi", :type "integer"}
         {:name "transmit-power", :type "integer"}
         {:name "max-transmit-power", :type "integer"}]}}]}
    {:id ::show-error-bubble,
     :name "showErrorBubble",
     :since "42",
     :params
     [{:name "message", :type "string"}
      {:name "link-range", :type "object"}
      {:name "link-target", :type "string"}
      {:name "anchor-rect", :type "object"}]}
    {:id ::hide-error-bubble, :name "hideErrorBubble", :since "43"}
    {:id ::set-auto-pairing-result,
     :name "setAutoPairingResult",
     :since "42",
     :callback? true,
     :params [{:name "result", :type "object"} {:name "callback", :optional? true, :type :callback}]}
    {:id ::find-setup-connection,
     :name "findSetupConnection",
     :since "47",
     :callback? true,
     :params
     [{:name "setup-service-uuid", :type "string"}
      {:name "time-out", :type "integer"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "connection-id", :type "integer"} {:name "device-address", :type "string"}]}}]}
    {:id ::setup-connection-status,
     :name "setupConnectionStatus",
     :since "47",
     :callback? true,
     :params
     [{:name "connection-id", :type "integer"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "status", :type "easyUnlockPrivate.ConnectionStatus"}]}}]}
    {:id ::setup-connection-disconnect,
     :name "setupConnectionDisconnect",
     :since "47",
     :callback? true,
     :params [{:name "connection-id", :type "integer"} {:name "callback", :optional? true, :type :callback}]}
    {:id ::setup-connection-send,
     :name "setupConnectionSend",
     :since "47",
     :callback? true,
     :params
     [{:name "connection-id", :type "integer"}
      {:name "data", :type "ArrayBuffer"}
      {:name "callback", :optional? true, :type :callback}]}
    {:id ::setup-connection-get-device-address,
     :name "setupConnectionGetDeviceAddress",
     :since "47",
     :callback? true,
     :params
     [{:name "connection-id", :type "integer"}
      {:name "callback", :type :callback, :callback {:params [{:name "device-address", :type "string"}]}}]}],
   :events
   [{:id ::on-user-info-updated,
     :name "onUserInfoUpdated",
     :params [{:name "user-info", :type "easyUnlockPrivate.UserInfo"}]}
    {:id ::on-start-auto-pairing, :name "onStartAutoPairing", :since "42"}
    {:id ::on-connection-status-changed,
     :name "onConnectionStatusChanged",
     :since "47",
     :params
     [{:name "connection-id", :type "integer"}
      {:name "old-status", :type "easyUnlockPrivate.ConnectionStatus"}
      {:name "new-status", :type "easyUnlockPrivate.ConnectionStatus"}]}
    {:id ::on-data-received,
     :name "onDataReceived",
     :since "47",
     :params [{:name "connection-id", :type "integer"} {:name "data", :type "ArrayBuffer"}]}
    {:id ::on-send-completed,
     :name "onSendCompleted",
     :since "47",
     :params
     [{:name "connection-id", :type "integer"}
      {:name "data", :type "ArrayBuffer"}
      {:name "success", :type "boolean"}]}]})

; -- helpers ----------------------------------------------------------------------------------------------------------------

; code generation for native API wrapper
(defmacro gen-wrap [kind item-id config & args]
  (let [static-config (get-static-config)]
    (apply gen-wrap-from-table static-config api-table kind item-id config args)))

; code generation for API call-site
(defn gen-call [kind item src-info & args]
  (let [static-config (get-static-config)
        config (gen-active-config static-config)]
    (apply gen-call-from-table static-config api-table kind item src-info config args)))