(ns auth.token
  (:require [clojure.java.io :as io]
            [buddy.core.keys :as ks]
            [buddy.sign.jws :as jws]
            [buddy.sign.util :as util]
            [clj-time.core :as t]
            [schema.core :as s]
            [cats.monad.exception :as ex]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Schemas

(def TokenConf
  {:privkey    s/Str
   :pubkey     s/Str
   :passphrase s/Str
   :token-exp  s/Int})

(def Token s/Str)

(def Payload
  {s/Keyword s/Any})

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(defn maybe-resource
  [^String path]
  {:pre  [(string? path)]
   :post [(and % (.exists (io/file %)))]}
  (if (.isAbsolute (io/file path))
    path
    (io/resource path)))

(defn private-key [auth-conf]
  (ks/private-key
    (maybe-resource (:privkey auth-conf))
    (:passphrase auth-conf)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public

(s/defn ^:always-validate create :- Token
  [payload :- Payload auth-conf :- TokenConf]
  (println "Creating token with payload" payload "expiring in" (:token-exp auth-conf))
  (let [exp (-> (t/plus (t/now) (t/seconds (:token-exp auth-conf))) (util/to-timestamp))]
    (jws/sign {:payload payload}
              (private-key auth-conf)
              {:alg :rs256 :exp exp})))

(s/defn ^:always-validate unsign :- Payload
  [token :- Token public-key-path :- String]
  (-> (jws/unsign token (ks/public-key (maybe-resource public-key-path)) {:alg :rs256})
      :payload))

(s/defn ^:always-validate expired? :- s/Bool
  [token :- Token public-key-path :- String]
  (let [result (jws/decode token (ks/public-key (maybe-resource public-key-path)) {:alg :rs256})]
      (cond
        (ex/success? result) false
        (and (ex/failure result) (= :exp (:cause result))) true
        :else @result)))                                    ; rethrow

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Examples

(comment
  (let [token (create {:username "xxx" :password "eee"}
                      {:privkey    "dev/auth_privkey.pem"
                       :passphrase "Pasword1"
                       :pubkey "dev/auth_pubkey.pem"
                       :token-exp 1})]
    (Thread/sleep 2000)
    (jws/decode token (ks/public-key (maybe-resource "dev/auth_pubkey.pem")) {:alg :rs256})))
