(ns auth.store
  (:require [yesql.core :refer [defqueries]]
            [clojure.edn :as edn]
            [clojure.string :as str]
            [buddy.core.hash :as hash]
            [buddy.core.codecs :as codecs])
  (:import (java.sql BatchUpdateException)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(defprotocol IUserStore
  (-add-user! [this user])
  (-find-user-by-username [this username])
  (-find-user-by-id [this id])
  (-destroy-user! [this user-id]))

(defprotocol IRefreshTokenStore
  (-add-refresh-token! [this user-id token])
  (-refresh-token-exists? [this user-id token])
  (-destroy-refresh-token! [this user-id token]))

(def resource-prefix "puri:auth:")

(defn add-resource-prefix
  [id]
  (str resource-prefix id))

(defn remove-resource-prefix
  [puri]
  (when puri
    (let [re-prefix (re-pattern resource-prefix)]
      (if (re-find re-prefix puri)
        (str/replace puri re-prefix "")
        (throw (ex-info (str "ID has to start with " resource-prefix)
                        {:type :validation
                         :cause :resource-id}))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Memory-based store (testing)

(deftype MemoryStore [users tokens]
  IUserStore
  (-add-user! [_ user]
    (let [id (add-resource-prefix (gensym))
          new-user (merge user
                          {:id id})]
      (swap! users assoc id new-user)
      new-user))
  (-find-user-by-username [_ username]
    (->> @users
         vals
         (filter #(= username (:username %)))
         first))
  (-find-user-by-id [_ id]
    (get @users id))
  (-destroy-user! [_ user-id]
    (let [existed? (some? (get @users user-id))]
      (swap! users dissoc user-id)
      (swap! tokens (fn [tokens]
                      (into #{} (remove #(= user-id (:user-id %)) tokens))))
      existed?))
  IRefreshTokenStore
  (-add-refresh-token! [_ user-id token]
    (swap! tokens conj {:user-id user-id :token token}))
  (-refresh-token-exists? [_ user-id token]
    (contains? @tokens {:user-id user-id :token token}))
  (-destroy-refresh-token! [_ user-id token]
    (swap! tokens disj {:user-id user-id :token token})))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Postgres-based store

(declare pg-add-user<! pg-find-user-by-username pg-find-user-by-id pg-destroy-user!)
(defqueries "sql/users.sql")
(declare pg-add-refresh-token! pg-find-refresh-token pg-destroy-refresh-token! pg-destroy-refresh-tokens!)
(defqueries "sql/refresh_tokens.sql")

(defn id->puri
  [id]
  (-> id
      str
      add-resource-prefix))

(defn puri->id
  [puri]
  (-> puri
      remove-resource-prefix
      edn/read-string))

(defn pg-record->user
  [record]
  (-> record
      (update :policy edn/read-string)
      (update :id id->puri)))

(defn user->pg-record
  [user]
  (-> user
      (update :policy prn-str)
      (update :id puri->id)))

(defn token->digest
  [token]
  (codecs/bytes->hex (hash/sha1 token)))

(defmacro unwrap-sql-exceptions
  [& body]
  `(try
     ~@body
     (catch BatchUpdateException e#
       (throw (or (.getNextException e#) e#)))))

(deftype PostgresStore [db]
  IUserStore
  (-add-user! [_ user]
    (unwrap-sql-exceptions
      (-> (apply pg-add-user<! db ((juxt :username :password :policy) (user->pg-record user)))
          pg-record->user)))
  (-find-user-by-username [_ username]
    (some-> (pg-find-user-by-username db username)
            first
            pg-record->user))
  (-find-user-by-id [_ id]
    (some-> (pg-find-user-by-id db (puri->id id))
            first
            pg-record->user))
  (-destroy-user! [_ user-id]
    (unwrap-sql-exceptions
      (pg-destroy-refresh-tokens! db (puri->id user-id))
      (> (pg-destroy-user! db (puri->id user-id)) 0)))
  IRefreshTokenStore
  (-add-refresh-token! [_ user-id token]
    (unwrap-sql-exceptions
      (pg-add-refresh-token! db (puri->id user-id) (token->digest token))))
  (-destroy-refresh-token! [_ user-id token]
    (unwrap-sql-exceptions
      (pg-destroy-refresh-token! db (puri->id user-id) (token->digest token))))
  (-refresh-token-exists? [_ user-id token]
    (not-empty (pg-find-refresh-token db (puri->id user-id) (token->digest token)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public

(defn memory-store
  []
  (MemoryStore. (atom {}) (atom #{})))

(defn pg-store
  [db]
  (PostgresStore. db))

(defn add-user!
  [ds user]
  (-add-user! ds user))

(defn find-user-by-username
  [ds username]
  (-find-user-by-username ds username))

(defn find-user-by-id
  [ds id]
  (-find-user-by-id ds id))

(defn destroy-user!
  [ds id]
  (-destroy-user! ds id))

(defn add-refresh-token!
  [ds user-id token]
  (-add-refresh-token! ds user-id token))

(defn destroy-refresh-token!
  [ds user-id token]
  (-destroy-refresh-token! ds user-id token))

(defn refresh-token-exists?
  [ds user-id token-]
  (-refresh-token-exists? ds user-id token-))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Examples

(comment
  (def db (pg-store (:database-url environ.core/env)))
  ;(def db (memory-store))
  (add-user! db {:username "bob1" :password "123" :policy {:version    1
                                                          :statements []}})
  (find-user-by-username db "bob1")
  (find-user-by-id db "puri:auth:7")
  (destroy-user! db "puri:auth:89")
  (add-refresh-token! db "puri:auth:89" "digest")
  (refresh-token-exists? db "puri:auth:89" "digest")
  (destroy-refresh-token! db "puri:auth:7" "digest"))