(ns auth.service
  (:require [auth.store :as store]
            [auth.token :as token]
            [buddy.hashers :as hashers]
            [schema.core :as s]
            [clojure.java.jdbc :as jdbc]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Schemas

(def Store s/Any)

(def User
  {(s/pred #(and (not= :password %) (keyword? %))) s/Any})

(def UserId
  s/Str)

(def UserWithId
  (merge User
         {:id UserId}))

(def NewUser
  {s/Keyword s/Any
   :password s/Str})

(def AuthCredentials
  (s/either {:username s/Str
             :password s/Str}
            {:refresh-token s/Str}))

(def AuthConf {:auth-policy s/Keyword
               s/Keyword    s/Any})

(def AuthResponse
  (s/either [(s/one s/Bool true) User]
            [(s/one s/Bool false) {:message s/Str}]))

(def Token s/Str)

(def TokenResponse
  (s/either [(s/one s/Bool true) {:token Token}]
            AuthResponse))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(defn create-auth-token
  [user auth-conf]
  (token/create user (:auth-token-conf auth-conf)))

(defn create-and-save-refresh-token!
  [ds user auth-conf]
  (let [token (token/create {:user-id (:id user)} (:refresh-token-conf auth-conf))]
    (store/add-refresh-token! ds (:id user) token)
    token))

(defn make-token-pair!
  [ds user auth-conf]
  {:token         (create-auth-token user auth-conf)
   :refresh-token (create-and-save-refresh-token! ds user auth-conf)})

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public

(s/defn ^:always-validate add-user! :- UserWithId
  [ds :- Store user :- NewUser]
  (-> (store/add-user! ds (update-in user [:password] #(hashers/encrypt %)))
      (dissoc :password)))

(s/defn ^:always-validate destroy-user! :- s/Bool
  [ds :- Store id :- UserId]
  (store/destroy-user! ds id))

(s/defn ^:always-validate auth-user :- AuthResponse
  [ds :- Store credentials :- AuthCredentials]
  (let [user (store/find-user-by-username ds (:username credentials))
        unauthed [false {:message "Invalid username or password"}]]
    (if user
      (if (hashers/check (:password credentials) (:password user)) ; This intentionally takes about 0.5s
        [true (dissoc user :password)]
        unauthed)
      unauthed)))

(defn- create-auth-tokens-dispatch-fn
  [_ds auth-conf credentials]
  [(:auth-policy auth-conf) (set (keys credentials))])

(defmulti create-auth-tokens create-auth-tokens-dispatch-fn)

(s/defmethod ^:always-validate create-auth-tokens [:auth-token #{:username :password}] :- TokenResponse
  [ds :- Store auth-conf :- AuthConf credentials :- AuthCredentials]
  (let [[ok? user :as auth-response] (auth-user ds credentials)]
    (if ok?
      [true {:token (create-auth-token user auth-conf)}]
      auth-response)))

(s/defmethod ^:always-validate create-auth-tokens [:refresh-token #{:username :password}] :- TokenResponse
  [ds :- Store auth-conf :- AuthConf credentials :- AuthCredentials]
  (let [[ok? user :as auth-response] (auth-user ds credentials)]
    (if ok?
      [true (make-token-pair! ds user auth-conf)]
      auth-response)))

(s/defmethod ^:always-validate create-auth-tokens [:refresh-token #{:refresh-token}] :- TokenResponse
  [ds :- Store auth-conf :- AuthConf credentials :- AuthCredentials]
  (let [rt (:refresh-token credentials)
        unsigned-rt (token/unsign rt (get-in auth-conf [:refresh-token-conf :pubkey]))
        user-id (:user-id unsigned-rt)
        user (and user-id (store/find-user-by-id ds user-id))]
    ;; TODO: Add transaction.
    (if (and (store/refresh-token-exists? ds user-id rt)
             user)
      (do (store/destroy-refresh-token! ds user-id rt)
          [true (make-token-pair! ds user auth-conf)])
      [false {:message "Refresh token invalid, revoked or expired"}])))

(defmethod create-auth-tokens :default
  [_ds auth-conf credentials]
  (throw (ex-info "Unsupported authentication policy" (merge (select-keys auth-conf [:auth-policy])
                                                             {:credentials (keys credentials)}))))
