(ns auth.policy
  (:require [schema.core :as s]
            [schema.coerce :as coerce]
            [schema.utils :as utils]
            [cheshire.core :as cheshire]
            [cheshire.generate :as generate]
            [clojure.string :as str])
  (:import [schema.utils ValidationError]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Schemas

(def Resource s/Str)

(def Action s/Str)

(def Statement
  {:actions   [Action]
   :resources [Resource]})

(def Policy
  {:version    s/Int
   :statements [Statement]})

(def AnyKeywordMap
  {s/Keyword s/Any})

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(def parse-policy* (coerce/coercer Policy coerce/json-coercion-matcher))

(generate/add-encoder
  ValidationError
  (fn [err jsonGenerator]
    (generate/encode-seq (utils/validation-error-explain err) jsonGenerator)))

(defn matches-pattern? [s pattern]
  (re-matches (-> pattern
                  (str/replace "*" ".*")
                  (re-pattern))
              s))

(defn matching-statement?
  [resource action statement]
  (and
    (not-empty (filter (partial matches-pattern? resource) (:resources statement)))
    (not-empty (filter (partial matches-pattern? action) (:actions statement)))))

(defn cart [colls]
  (if (empty? colls)
    '(())
    (for [x (first colls)
          more (cart (rest colls))]
      (cons x more))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public

(defn parse-policy
  "Corces a policy (e.g. one read from JSON with already keywordized keys) and validates it."
  [policy]
  (let [result (parse-policy* policy)]
    (if-not (utils/error? result)
      result
      (throw (ex-info "Malformed or invalid policy" {:error (utils/error-val result)})))))

(s/defn ^:always-validate json->policy :- Policy
  "Converts a JSON string to policy throwing an error if its incorrect."
  [json :- String]
  (parse-policy (cheshire/parse-string json true)))

(s/defn ^:always-validate policy->json :- String
  "Converts a policy to JSON string."
  [policy :- Policy]
  (cheshire/generate-string policy))

(defmulti deny-action?
  "Dispatched on action. Return true to deny an action, false to defer to the built-in verification."
  (fn [_ action & _] action))

(defmethod deny-action? :default
  [_resource _action _policy & [_opts]]
  false)

(s/defn ^:always-validate allowed? :- s/Bool
  "Returns true if the resource/action combination is allowed under the specified policy.

   You can specify an optional map, `opts` that will be passed to [[deny-action?]]; see [[auth.handler/create-user!]]
   and [[auth.handler/deny-action?] for example usage."
  [policy :- Policy resource :- Resource action :- Action & [opts] :- [(s/maybe AnyKeywordMap)]]
  (and (not (deny-action? resource action policy opts))
       (not (empty? (filter #(matching-statement? resource action %) (:statements policy))))))

(defn dbg--
  [& args]
  ;(apply println args)
  (last args))

(s/defn ^:always-validate subset? :- s/Bool
  "Returns true if `policy1` is a subset of `policy2`."
  [policy1 :- Policy policy2 :- Policy]
  (->> policy1
       ;(dbg-- "policy1")
       :statements
       ;(dbg-- "statements")
       (map (juxt :resources :actions))
       (mapcat cart)
       ;(dbg-- "cart =")
       (filter (fn [[r a]] (not (allowed? policy2 r a))))
       ;(dbg "filter =")
       empty?))

