(ns auth.middleware
  (:require [auth.token :as token]
            [auth.policy :as policy]
            [auth.header :as header]
            [environ.core :refer [env]]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(defn token-from-headers
  [req]
  (let [s (->> (get-in req [:headers "authorization"])
               (auth.header/parse-authorization-header))]
    (when (= "DLY-TOKEN" (:auth-scheme s))
      (:token s))))

(defn token-from-params
  [req]
  (or (get-in req [:params :token])
      (get-in req [:params "token"])))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public

(defn authenticated? [req]
  (some? (get-in req [:auth-user :policy])))

(defn ex->msg
  [ex default]
  (or (:message (ex-data ex)) (.getMessage ex) default))

(defn ex->response
  [e]
  (if-let [data (ex-data e)]
    (let [tc ((juxt :type :cause) data)
          response (case tc
                     [:validation :exp] {:status 401 :body (ex->msg e "Invalid authentication token")}
                     [:validation :schema.core/error] {:status 400 :body "Malformed request"}
                     [:authorization :restrict] {:status 401 :body (ex->msg e "Unavailable")}
                     [:validation :resource-id] {:status 400 :body "Malformed resource ID"}
                     {:status 400 :body (ex->msg e "Bad request")})]
      response)
    {:status 500 :body (.getMessage e)}))

(defn wrap-exceptions
  [handler]
  (fn [req]
    (try
      (handler req)
      (catch Exception e
        (ex->response e)))))                                ;; TODO: Log.

(defn wrap-auth-token-params
  [handler & [pubkey-path]]
  (fn [req]
    (let [user (when-let [token (or (token-from-headers req)
                                    (token-from-params req))]
                 (token/unsign token (or pubkey-path (get-in env [:pubkey-path]))))] ;; Duplicated in AuthConf.
      (handler (assoc req :auth-user user)))))

(defn wrap-restrict
  [handler resource action & [opts]]
  (fn [req]
    (let [auth-user (:auth-user req)]
      (if (and auth-user
               (:policy auth-user)
               (policy/allowed? (:policy auth-user) resource action opts))
        (handler req)
        {:status 401 :body {:message "Unauthorized"}}))))

(defn restrict!
  [resource action req]
  (let [auth-user (:auth-user req)]
    (when-not (and auth-user
                   (:policy auth-user)
                   (policy/allowed? (:policy auth-user) resource action))
      (throw (ex-info "Unauthorized" {:type :authorization :cause :restrict})))))

