(ns auth.handler
  (:require [auth.service :as service]
            [auth.policy :as policy]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(defn allowed?
  [auth-user resource action & [opts]]
  (and auth-user
       (:policy auth-user)
       (policy/allowed? (policy/parse-policy (:policy auth-user)) resource action opts)))

;; Make it impossible to create user with a wider policy.
(defmethod policy/deny-action? "auth:CreateUser"
  [_resource _action policy & [opts]]
  (let [new-user (::new-user opts)
        new-user-policy (:policy new-user)]
    (not (or (empty? new-user-policy)
             (policy/subset? new-user-policy policy)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Handlers

(defn create-auth-token [{:keys [datastore auth-conf params]}]
  (let [[ok? res] (service/create-auth-tokens datastore auth-conf params)]
    (if ok?
      {:status 201 :body res}
      {:status 401 :body res})))

(defn create-user! [{:keys [datastore auth-user] {new-user :user} :params}]
  (if (allowed? auth-user "*" "auth:CreateUser" {::new-user new-user})
    (let [user (service/add-user! datastore new-user)]
      (assert (nil? (:password user)))
      {:status 200 :body {:user user}}) 
    {:status 401 :body {:message "Unauthorized"}}))

(defn destroy-user! [{:keys [datastore auth-user] {id :id} :params}]
  (if (allowed? auth-user id "auth:DestroyUser")
    {:status 200 :body {:existed? (service/destroy-user! datastore id)}}
    {:status 401 :body {:message "Unauthorized"}}))
