(ns auth.core
  (:require [auth.store :as store]
            [auth.service :as service]
            [auth.handler :as handler]
            [auth.middleware :as middleware]
            [auth.db.migrations :as migrations]
            [compojure.core :refer [defroutes POST DELETE GET]]
            [compojure.route :refer [not-found]]
            [ring.middleware.keyword-params :refer [wrap-keyword-params]]
            [ring.middleware.json :refer [wrap-json-response wrap-json-params]]
            [environ.core :refer [env]]
            [org.httpkit.server :as http-kit]
            [ring.middleware.cors :refer [wrap-cors]
             ])
  (:gen-class))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Implementation

(defroutes api-routes
  (POST "/auth-token" [] handler/create-auth-token)
  (POST "/users" [] handler/create-user!)
  (DELETE "/users/:id" [] handler/destroy-user!)
  ;(compojure.core/rfn req
  ;  (println req)
  ;  {:status 404 :body (str "Not found " (prn-str req))})
  (not-found "Not found"))

(defn get-int
  [env k default]
  (or (when-let [s (get env k)] (Integer/parseInt s))
      default))

(defn wrap-datasource
  [handler ds]
  (fn [req]
    (handler (assoc req :datastore ds))))

(defn wrap-config
  [handler]
  (fn [req]
    (handler (assoc req :auth-conf {:auth-policy        :refresh-token
                                    :auth-token-conf    {:privkey    (get env :privkey-path)
                                                         :passphrase (get env :privkey-passphrase)
                                                         :pubkey     (get env :pubkey-path)
                                                         :token-exp  (get-int env :token-exp (* 24 60 60))}
                                    :refresh-token-conf {:privkey    (get env :privkey-path)
                                                         :passphrase (get env :privkey-passphrase)
                                                         :pubkey     (get env :pubkey-path)
                                                         :token-exp  (get-int env :refresh-token-exp (* 365 24 60 60))}}))))

(defn ignore-trailing-slash
  "Modifies the request uri before calling the handler.
  Removes a single trailing slash from the end of the uri if present.

  Useful for handling optional trailing slashes until Compojure's route matching syntax supports regex.
  Adapted from http://stackoverflow.com/questions/8380468/compojure-regex-for-matching-a-trailing-slash"
  [handler]
  (fn [request]
    (let [uri (:uri request)]
      (handler (assoc request :uri (if (and (not (= "/" uri))
                                            (.endsWith uri "/"))
                                     (subs uri 0 (dec (count uri)))
                                     uri))))))

(defn app-routes
  [ds]
  (-> api-routes
      ignore-trailing-slash
      middleware/wrap-auth-token-params
      (wrap-datasource ds)
      wrap-config
      wrap-keyword-params
      middleware/wrap-exceptions
      wrap-json-params
      wrap-json-response
      (wrap-cors :access-control-allow-origin #"http://localhost:3000"
                 :access-control-allow-methods [:get :put :post])))

(defn get-ds
  []
  (store/pg-store (:database-url env)))

(def app (app-routes (get-ds)))

(defn seed [datastore [_ user-name password]]
  (assert user-name "User name is required")
  (assert password "Password is required")
  (when-not (store/find-user-by-username datastore user-name)
    (println "Adding" user-name)
    (service/add-user! datastore {:username user-name
                                  :password password
                                  :policy   {:version    1,
                                             :statements [{:actions   ["auth:CreateUser" "auth:DestroyUser"
                                                                       "monitoring:CreateAgent" "monitoring:DestroyAgent" "monitoring:GetEvents"
                                                                       "subscriptions:CreateSubscription" "subscriptions:DestroySubscription"
                                                                       "subscriptions:OpenSubscription" "subscriptions:CloseSubscription"
                                                                       "subscriptions:PutResources"
                                                                       "storage:CreateFolder" "storage:DestroyFolder" "storage:GetFolder"],
                                                           :resources ["*"]}
                                                          {:actions   ["subscriptions:GetSubscription",
                                                                       "monitoring:PutLog", "monitoring:PutEvent",
                                                                       "storage:GetCredentials", "storage:DeleteCredentials"]
                                                           :resources ["*"]}]}})))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Server


(defonce server (atom nil))

(defn parse-port [[port]]
  (Integer/parseInt (or port (env :port) "3000")))

(defn start-server [port]
  ;(init)
  (reset! server
          (http-kit/run-server
            app
            {:port port})))

(defn stop-server []
  (when @server
    ;(destroy)
    (@server :timeout 100)
    (reset! server nil)))

(defn start-app [args]
  (let [port (parse-port args)]
    (.addShutdownHook (Runtime/getRuntime) (Thread. stop-server))
    ;(timbre/info "server is starting on port " port)
    (println "Server is starting on port" port)
    (start-server port)))

(defn -main [& args]
  (cond
    (some #{"migrate" "rollback"} args) (migrations/migrate args)
    (some #{"seed"} args) (seed (get-ds) args)
    :else (start-app args)))

(comment
  (store/destroy-user! (get-ds) (:id (store/find-user-by-username (get-ds) "33")))
  (service/add-user! (get-ds) {:username "90" :password "90"
                               :policy   {:version    1
                                          :statements [{:actions   ["subscriptions:GetSubscription"]
                                                        :resources ["puri:subscriptions:90"]}]}})
  (service/add-user! (get-ds) {:username "91" :password "91"
                               :policy   {:version    1
                                          :statements [{:actions   ["subscriptions:GetSubscription"]
                                                        :resources ["puri:subscriptions:91"]}]}})
  (seed (get-ds))
  (start-server 3001)
  (stop-server))
