(ns auth.api-conn
  (:require [auth.header :as header]
            [clj-http.client :as http]))

;; Workaround for https://github.com/dakrone/clj-http/issues/257
(defmethod http/coerce-response-body :clojure [req resp]
  (let [cl (get-in resp [:headers "Content-Length"])]
    (if (or (nil? cl) (> (Integer/parseInt cl)))
      (http/coerce-clojure-body req resp)
      (assoc resp :body nil))))


(defn request*
  [method url params & [headers opts]]
  (let [res (http/request (merge {:method           method
                                  :url              url
                                  :content-type     :json
                                  :headers          headers
                                  :throw-exceptions false
                                  :as               :json
                                  :form-params      params}
                                 opts))]
    res))

(defn request-error!
  [msg cause res]
  (throw (ex-info msg {:response res
                       :type     ::request-error
                       :cause    cause})))

(defprotocol IAuthenticate
  (-authenticate [this]))

(defprotocol IApiRequest
  (-request [this method path params]))

(declare authenticate*)

(deftype AuthClient [auth-endpoint credentials tokens]
  IApiRequest
  (-request [_ method path params]
    (request* method (str auth-endpoint path) params))
  IAuthenticate
  (-authenticate [this]                                     ;; TODO: Monads.
    (let [refresh-token (:refresh-token tokens)
          [success res] (and refresh-token (authenticate* this {:refresh-token refresh-token}))]
      (if success
        res
        (let [[success res] (authenticate* this credentials)]
          (if success
            res
            (request-error! "Authentication failed" ::authenticate res)))))))

(defn authenticate*
  [auth-client credentials]
  (let [res (-request auth-client :post "/auth-token/" credentials)]
    (if (http/success? res)
      [true (AuthClient. (.-auth-endpoint auth-client) (.-credentials auth-client) (select-keys (:body res) [:token :refresh-token]))]
      [false res])))

(defn auth-headers
  [auth-client]
  {"Authorization" (header/make-authorization-header "DLY-TOKEN" (select-keys (.-tokens auth-client) [:token]))})

(defn serialize-auth-client
  [auth-client]
  {:auth-endpoint (.-auth-endpoint auth-client)
   :credentials   (.-credentials auth-client)
   :tokens        (.-tokens auth-client)})

(defn deserialize-auth-client
  [edn]
  (AuthClient. (:auth-endpoint edn) (:credentials edn) (:tokens edn)))

(defn make-auth-client [endpoint credentials & [tokens]]
  (AuthClient. endpoint credentials tokens))

(deftype ApiConn [auth-client http-options]
  IApiRequest
  (-request [_ method url params]
    (let [request-fn (fn [auth-client] (request* method url params (auth-headers auth-client)
                                                 http-options))
          first-res (request-fn @auth-client)
          res (cond (http/success? first-res) first-res
                    (contains? #{400 401 403} (:status first-res)) (do (reset! auth-client (-authenticate @auth-client))
                                                                   (request-fn @auth-client))
                    :else first-res)]
      (if (http/success? res)
        res
        (request-error! (str "Request to " url " failed") ::api-request res)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public

(defn make-api-conn [{:keys [auth-endpoint credentials tokens http-options]}]
  (let [auth (-authenticate (make-auth-client auth-endpoint credentials tokens))]
    (ApiConn. (atom auth) http-options)))

(defn request!
  [api method path params]
  (:body (-request api method path params)))

(defn serialize
  [api-conn]
  {:auth-client  (serialize-auth-client @(.-auth-client api-conn))
   :http-options (.http_options api-conn)})

(defn deserialize
  [edn]
  (ApiConn. (atom (deserialize-auth-client (:auth-client edn))) (:http-options edn)))

(defn token
  [api-conn]
  (-> api-conn
      .-auth-client
      deref
      .-tokens
      :token))

(defn refresh-token
  [api-conn]
  (-> api-conn
      .-auth-client
      deref
      .-tokens
      :refresh-token))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Examples

(comment
  (def credentials {:username "91" :password "91"})
  (def credentials {:username "test" :password "secret"})
  (def sub-api (make-api-conn {:auth-endpoint "http://localhost:3000"
                               :credentials   credentials
                               :http-options  {:content-type :edn
                                               :accept       :edn
                                               :as :clojure}}))
  (def sub-api (make-api-conn {:auth-endpoint "https://proton.red.designed.ly/auth/api/1"
                               :credentials   credentials
                               :http-options  {:content-type :edn
                                               :accept       :edn
                                               :as :clojure}}))
  
  (request! sub-api :get "http://localhost:8080/subscriptions/api/2/subscriptions/puri:subscriptions::90" {})
  (request! sub-api :get "http://localhost:8080/subscriptions/api/2/subscriptions/?activation-key=3e08a5b8-7320-50ba-a4a4-db4228855c64" {})

  (request! sub-api :get "http://localhost:8080/monitoring/api/2/agents/puri:monitoring::70/logs/fname2" {})
  (request! sub-api :get "http://localhost:8080/monitoring/api/2/agents/puri:monitoring::70/logs/" {})
  (request! sub-api :get "https://proton.red.designed.ly/monitoring/api/2/agents/puri:monitoring::3/logs/" {})

  )
