(ns beam-aws.core
  (:require [amazonica.aws.s3 :as s3]
            [amazonica.aws.dynamodbv2 :as dynamo]
            [amazonica.aws.cognitoidp :as cognito]
            [amazonica.aws.lambda :as lambda]
            [cheshire.core :as json]
            [camel-snake-kebab.core :refer :all]
            [camel-snake-kebab.extras :refer [transform-keys]]
            [clojure.java.io :as io]
            [taoensso.timbre :as log]
            [environ.core :as environ])
  (:import (com.amazonaws.services.dynamodbv2.model ProvisionedThroughputExceededException)))

; ==============================================================================

; To use this aws library of functions, you must have your aws credentials set
; via environment variables.
;
; You need these variables set:
;
;  AWS_ACCESS_KEY_ID – AWS access key.
;  AWS_SECRET_ACCESS_KEY – AWS secret key. Access and secret key variables
; override credentials stored in credential and config files.
;  AWS_DEFAULT_REGION – AWS region. This variable overrides the default region of
; the in-use profile, if set.

; ==============================================================================


(def sleep-ms 200)

; ==============================================================================
; LAMBDA FUNCTIONS
; ==============================================================================

(defn- parse-lambda-response
  "Parse the lambda buffer response"
  [lambda-reponse-future]
  (let [{:keys [status-code payload] :as lambda-response} lambda-reponse-future
        payload (new java.lang.String (.array payload) "UTF-8")]
    (if (= 200 status-code)
      (json/parse-string payload true)
      (log/error "Invoke lambda call failed"))))

(defn invoke-lambda!
  "Invokes a lambda function based on function-name (string) and payload (a map)."
  ([function-name]
   (log/info "Invoked " function-name " with no payload")
   (->> (lambda/invoke :function-name function-name)
        parse-lambda-response))
  ([function-name payload]
   (log/info "Invoked " function-name " with payload: " payload)
   (let [payload-string (json/generate-string payload)]
     (->> payload-string
          (lambda/invoke :function-name function-name
                         :payload ,,,)
          parse-lambda-response))))

; ==============================================================================
; USER/COGNITO FUNCTIONS
; ==============================================================================

(defn list-user-pools [] (cognito/list-user-pools {:max-results 2}))

(defn admin-create-user!
  "Creates a user in Cognito. Attributes in map format."
  [user-pool-id username attributes]
  (cognito/admin-create-user :user-pool-id user-pool-id
                             :username username))

(defn- aws-attribute-map
  "Creates a string suitable for aws `user-attributes` from an attribute map."
  [attributes]
  (let [keynames (map name (keys attributes))
        values (vals attributes)]
    (map #(hash-map "Name" %1 "Value" %2) keynames values)))

(defn- parse-aws-attributes
  "Creates an attribute map from AWS attribute map."
  [aws-attributes]
  (reduce (fn [result val] (assoc result (keyword (:name val)) (:value val))) {} aws-attributes))

(defn sign-up!
  "Creates a user in Cognito. Optional attributes provided in map format."
  ([client-id username password]
   (log/info "Signing up user " username)
   (cognito/sign-up :client-id client-id
                    :username username
                    :password password))
  ([client-id username password user-attributes]
   (log/info "Signing up user " username " with attributes " user-attributes)
   (cognito/sign-up :client-id client-id
                    :username username
                    :password password
                    :user-attributes (aws-attribute-map user-attributes))))

(defn initiate-auth!
  "Initiate authentication with Cognito"
  [client-id user-pool-id username password]
  (cognito/admin-initiate-auth :client-id client-id
                               :user-pool-id user-pool-id
                               :auth-flow "ADMIN_NO_SRP_AUTH"
                               :auth-parameters {"USERNAME" username "PASSWORD" password}))

(defn user!
  "Gets the user details from Cognito"
  [access-token]
  (log/info "Get user from Cognito")
  (let [{username :username user-attributes :user-attributes} (cognito/get-user :access-token access-token)]
    (merge {:username username} (parse-aws-attributes user-attributes))))

(defn admin-get-user!
  "Gets the user details from cognito without need for access token"
  [user-pool-id username]
  (log/info "Getting user " username " from Cognito")
  (let [{username :username user-attributes :user-attributes} (cognito/admin-get-user :user-pool-id user-pool-id
                                                                                      :username username)]
    (merge {:username username} (parse-aws-attributes user-attributes))))

(defn admin-update-user-attributes!
  "Updates the user's attributes - some or all. Attributes as a map."
  [user-pool-id username attributes]
  (cognito/admin-update-user-attributes :user-pool-id user-pool-id
                                        :username username
                                        :user-attributes (aws-attribute-map attributes)))

(defn admin-confirm-signup!
  "Verifies a user in Cognito"
  [user-pool-id email]
  (cognito/admin-confirm-sign-up :user-pool-id user-pool-id
                                 :username email))

(defn admin-delete-user!
  "Deletes user from cognito based on username"
  [user-pool-id username]
  (cognito/admin-delete-user :user-pool-id user-pool-id
                             :username username))

; ==============================================================================
; S3 FUNCTIONS
; ==============================================================================

(defn s3-delete-object!
  "Deletes an object from s3"
  [bucket path]
  (log/info "Deleting object located here " path)
  (s3/delete-object :bucket-name bucket
                    :key path))

(defn s3-list-objects!
  "Lists objects in s3bucket"
  [bucket prefix]
  (log/info "Requesting keys for " bucket "/" prefix)
  (loop [{:keys [truncated? next-marker object-summaries]}
         (s3/list-objects :bucket-name bucket
                          :prefix prefix)
         result []]
    (if (not truncated?)
      (into result object-summaries)
      (recur (s3/list-objects :bucket-name bucket
                              :prefix prefix
                              :marker next-marker) (into result object-summaries)))))

(defn s3-list-objects-keys!
  "List all keys below prefix"
  [bucket prefix]
  (map :key (s3-list-objects! bucket prefix)))

(defn s3-get-json!
  "Given an s3 bucket and key, will return a map of the json"
  [bucket key]
  (-> (s3/get-object bucket key)
      :input-stream
      io/reader
      (json/parse-stream true)))

(defn s3-get-camel-json!
  "Returns the json object converting the resutl from camel to kebab case"
  [bucket key]
  (->> (s3-get-json! bucket key) (transform-keys ->kebab-case ,,,)))

(defn s3-put-string!
  "PUT data to s3 bucket. Data is a string."
  [bucket path data]
  (log/info "Putting data to " path)
  (let [data-bytes (.getBytes data)]
    (with-open [is (io/input-stream data-bytes)]
      (s3/put-object :bucket-name bucket
                     :key path
                     :input-stream is
                     :metadata {:content-length (count data-bytes)}))))

(defn s3-put-json!
  "PUT this data as json to to the path in the S3 bucket"
  [bucket path data]
  (log/info "Putting data to " path)
  (let [json-bytes (.getBytes (json/generate-string data))]
    (with-open [is (io/input-stream json-bytes)]
      (s3/put-object :bucket-name bucket
                     :key path
                     :input-stream is
                     :metadata {:content-length (count json-bytes)}))))

(defn s3-put-camel-json!
  "PUT this data as json to to the path in the S3 bucket"
  [bucket path data]
  (log/info "Putting camel json data to" bucket "/" path)
  (let [json-bytes (.getBytes (json/generate-string data {:key-fn ->camelCaseString}))]
    (with-open [is (io/input-stream json-bytes)]
      (s3/put-object :bucket-name bucket
                     :key path
                     :input-stream is
                     :metadata {:content-length (count json-bytes)}))))

; ==============================================================================
; DYNAMO FUNCTIONS
; ==============================================================================

(defn dynamo-delete-key!
  "Deletes all key from dynamo table"
  [table-name key]
  (let [key (transform-keys ->camelCaseKeyword key)]
    (dynamo/delete-item
      :table-name table-name
      :key key)))

(defn dynamo-scan!
  "Scan table-name for items based on attribute value. Only returns the first 1MB
  of data (pre-filter size). This function is multi-arity. If called with
  last-evaluated-key, it will return as many items as possible from this point
  forward."
  ([table-name attribute attribute-type value attributes]
   (let [projection-expression (->> (map ->camelCaseString attributes)
                                    (clojure.string/join ", ",,,))
         column (->camelCaseString attribute)]
     (dynamo/scan
       :table-name table-name
       :projection-expression projection-expression
       :filter-expression "#key_placeholder = :value_placeholder"
       :expression-attribute-names {"#key_placeholder" column}
       :expression-attribute-values {":value_placeholder" {attribute-type value}})))
  ([table-name column column-type value attributes last-evaluated-key]
   (let [projection-expression (->> (map ->camelCaseString attributes)
                                    (clojure.string/join ", ",,,))
         column (->camelCaseString column)]
     (dynamo/scan
       :table-name table-name
       :projection-expression projection-expression
       :filter-expression "#key_placeholder = :value_placeholder"
       :expression-attribute-names {"#key_placeholder" column}
       :expression-attribute-values {":value_placeholder" {column-type value}}
       :exclusive-start-key last-evaluated-key))))

(defn dynamo-scan-all!
  "Scan table-name for items based on attribute value.
  Scans dynamo table and filters result based on attribute = value.
  Returns map of attributes (list of keynames) and value. Will loop through scan
  until all matching values from table are returned."
  [table-name attribute attribute-type value attributes]
  (loop [{:keys [items last-evaluated-key]} (dynamo-scan! table-name attribute attribute-type value attributes)
         result []]
    (if (nil? last-evaluated-key)
      (into result items)
      (recur (dynamo-scan! table-name attribute attribute-type value attributes last-evaluated-key) (into result items)))))

(defn dynamo-query!
  "Scan table-name for items based on attribute value. Only returns the first 1MB
  of data (pre-filter size). This function is multi-arity. If called with
  last-evaluated-key, it will return as many items as possible from this point
  forward."
  ([table-name partition-key partition-key-type value attributes]
   (let [projection-expression (->> (map ->camelCaseString attributes)
                                    (clojure.string/join ", ",,,))
         key-condition-expression (str (->camelCaseString partition-key) " = :value_placeholder")
         column (->camelCaseString partition-key)]
     (dynamo/query
       :table-name table-name
       :projection-expression projection-expression
       :key-condition-expression key-condition-expression
       :expression-attribute-values {":value_placeholder" {partition-key-type value}})))
  ([table-name partition-key partition-key-type value attributes last-evaluated-key]
   (let [projection-expression (->> (map ->camelCaseString attributes)
                                    (clojure.string/join ", ",,,))
         key-condition-expression (str (->camelCaseString partition-key) " = :value_placeholder")
         column (->camelCaseString partition-key)]
     (dynamo/query
       :table-name table-name
       :projection-expression projection-expression
       :key-condition-expression key-condition-expression
       :expression-attribute-values {":value_placeholder" {partition-key-type value}}
       :exclusive-start-key last-evaluated-key))))

(defn dynamo-query-all!
  "Scan table-name for items based on attribute value.
  Scans dynamo table and filters result based on attribute = value.
  Returns map of attributes (list of keynames) and value. Will loop through scan
  until all matching values from table are returned."
  [table-name attribute attribute-type value attributes]
  (loop [{:keys [items last-evaluated-key]} (dynamo-query! table-name attribute attribute-type value attributes)
         result []]
    (if (nil? last-evaluated-key)
      (into result items)
      (recur (dynamo-query! table-name attribute attribute-type value attributes last-evaluated-key) (into result items)))))

(defn dynamo-delete-device-ids!
  "Deletes all device-id items from table"
  [table-name device-id]
  (let [items (dynamo-query-all! table-name :device-id :s device-id [:device-id :id])]
    (doall (map #(dynamo-delete-key! table-name %) items))))

(defn dynamo-put-camel-item!
  "Puts a new item to dynamodb"
  [table-name {:keys [id device-id] :as item}]
  (log/info "Writing item: id: " id " device-id: " device-id " to " table-name " in dynamodb")
  (let [camel-cased-item (transform-keys ->camelCaseKeyword item)]
    (try
      (dynamo/put-item
        :table-name table-name
        :return-consumed-capacity "TOTAL"
        :return-item-collection-metrics "SIZE"
        :item camel-cased-item)
      (catch ProvisionedThroughputExceededException e
        (let [sleep-time (+ sleep-ms (rand-int sleep-ms))]
          (log/info "ProvisionedThroughputExceededException caught. Retrying after " sleep-time " ms.")
          (Thread/sleep sleep-time)
          (dynamo-put-camel-item! table-name item))))))

(defn dynamo-batch-put-request!
  "Submits batch put request as a future to dynamo"
  [put-request]
  (future
    (try
      (dynamo/batch-write-item
        :return-consumed-capacity "TOTAL"
        :return-item-collection-metrics "SIZE"
        :request-items put-request)
      (catch ProvisionedThroughputExceededException e
        (let [sleep-time (+ sleep-ms (rand-int sleep-ms))]
          (log/info "ProvisionedThroughputExceededException caught. Retrying after " sleep-time " ms.")
          (Thread/sleep sleep-time)
          (dynamo-batch-put-request! put-request))))))

(defn dynamo-batch-put-items!
  "Puts new items to dynamodb and loops until all items have been successfully
  put to dynamo."
  [table-name items]
  (loop [put-request {table-name (mapv (fn [item] {:put-request {:item item}}) items)}
         {:keys [unprocessed-items] :as request-response} @(dynamo-batch-put-request! put-request)]
    (if unprocessed-items
      (recur unprocessed-items @(dynamo-batch-put-request! put-request))
      request-response)))

(defn dynamo-batch-put-camel-items!
  "Puts new items to dynamodb by calling `batch-put-items-dynamo!`
  with each 25 item partition."
  [table-name items]
  (log/info "Batch writing " (count items) " items to " table-name " in dynamodb")
  (let [camel-cased-items (map #(transform-keys ->camelCaseKeyword %) items)
        partitioned-items (partition-all 25 camel-cased-items)]
    (doall (map #(dynamo-batch-put-items! table-name %) partitioned-items))))

(defn dynamo-batch-delete-device-ids!
  "Deletes all device-id items from table"
  [table-name device-id]
  (let [partitioned-items (partition-all 25 (dynamo-query-all! table-name :device-id :s device-id [:device-id :id]))]
    (doall (map (fn [partition]
                  (let [delete-request {table-name (mapv (fn [item] {:delete-request {:key item}}) partition)}]
                    (future (dynamo-batch-put-request! delete-request)))) partitioned-items))))

(defn dynamo-put-camel-items!
  "Puts a new item to dynamodb"
  [table-name items]
  (log/info "Writing " (count items) " items to " table-name " in dynamodb")
  (doall (pmap #(dynamo-put-camel-item! table-name %) items)))

(defn delete-camel-item-from-dynamo!
  "Deletes all entries with a particular device-id"
  [table-name hash-key]
  (dynamo/delete-item
    :table-name table-name
    :key hash-key))