(ns clue.template-test
  (:require [cljs.test :refer-macros [deftest is run-tests testing]]
            [clue.template :as ct]))

(deftest interpolate
  (testing "should take a value and returns a interpolation of it"
    (is (= "{{count}}" (ct/interpolate :count)))
    (is (= "{{count}}" (ct/interpolate "count")))
    (is (= "{{list}}" (ct/interpolate :list)))))

(deftest ->directive
  (testing "should take a directive and a key, and returns the parsed directive as keyword"
    (is (= :v-on:click (ct/->directive "v-on" :click)))
    (is (= :v-on:drag (ct/->directive "v-on" :drag)))
    (is (= :v-bind:value (ct/->directive "v-bind" :value)))))

(deftest on
  (testing "should take a event name and return the directive to it"
    (is (= :v-on:click (ct/on :click)))
    (is (= :v-on:drag (ct/on :drag)))
    (is (= :v-on:bind (ct/on :bind)))))

(deftest bind
  (testing "should take name and return the bind directive to it"
    (is (= :v-bind:click (ct/bind :click)))
    (is (= :v-bind:drag (ct/bind :drag)))
    (is (= :v-bind:bind (ct/bind :bind)))))

(deftest with-params
  (testing "should take a function name and the additional parameters names, and returns a function call string"
    (is (= "inc(a, b)" (ct/with-params :inc :a :b)))
    (is (= "inc(a, b, $event)" (ct/with-params :inc :a :b "$event")))
    (is (= "inc(a, b, $event)" (ct/with-params :inc :a :b :$event)))))

(deftest event
  (testing "should be a valid vue event"
    (is (= "$event" ct/event))))

(deftest count
  (testing "should take a value and returns the js version of getting the count as string"
    (is (= "(coll.length)" (ct/count :coll)))
    (is (= "(coll.length)" (ct/count "coll")))
    (is (= "(list.length)" (ct/count :list)))))

(deftest op2*
  (testing "should take an operation string, a value and an optional second value and returns the js version of the operation"
    (is (= "(a)" (ct/op2* "&&" :a)))
    (is (= "(a && b)" (ct/op2* "&&" :a :b)))
    (is (= "(a || b)" (ct/op2* "||" :a "b")))))

(deftest op*
  (testing "should take an operation an a sequence of values and returns the js version of the operation"
    (is (= "(true)" (ct/op* "&&")))
    (is (= "(a)" (ct/op* "&&" :a)))
    (is (= "(a && b)" (ct/op* "&&" :a :b)))
    (is (= "((a && b) && c)" (ct/op* "&&" :a :b :c)))
    (is (= "((a || b) || c)" (ct/op* "||" :a :b :c)))))

(deftest not
  (testing "should take a value and negate it"
    (is (= "!(true)" (ct/not true)))
    (is (= "(false)" (ct/not)))
    (is (= "!(a)" (ct/not :a)))
    (is (= "!(a)" (ct/not "a")))))

(deftest and
  (testing "should take a sequence of values and return the js version of and"
    (is (= "(a)" (ct/and :a)))
    (is (= "(true)" (ct/and)))
    (is (= "(a && b)" (ct/and :a "b")))
    (is (= "((a && b) && c)" (ct/and :a :b :c)))))

(deftest or
  (testing "should take a sequence of values and return the js version of or"
    (is (= "(a)" (ct/or :a)))
    (is (= "(true)" (ct/or)))
    (is (= "(a || b)" (ct/or :a "b")))
    (is (= "((a || b) || c)" (ct/or :a :b :c)))))

(deftest eq?
  (testing "should take a sequence of values and return the js version of =="
    (is (= "(a)" (ct/eq? :a)))
    (is (= "(true)" (ct/eq?)))
    (is (= "(a == b)" (ct/eq? :a :b)))
    (is (= "((a == b) && (b == c))" (ct/eq? :a :b :c)))
    (is (= "(((a == b) && (b == c)) && (c == d))" (ct/eq? :a :b :c :d)))))

(deftest gt?
  (testing "should take a sequence of values and return the js version of >"
    (is (= "(a)" (ct/gt? :a)))
    (is (= "(true)" (ct/gt?)))
    (is (= "(a > b)" (ct/gt? :a :b)))
    (is (= "((a > b) && (b > c))" (ct/gt? :a :b :c)))))

(deftest lt?
  (testing "should take a sequence of values and return the js version of <"
    (is (= "(a)" (ct/lt? :a)))
    (is (= "(true)" (ct/lt?)))
    (is (= "(a < b)" (ct/lt? :a :b)))
    (is (= "((a < b) && (b < c))" (ct/lt? :a :b :c)))))

(deftest eq-gt?
  (testing "should take a sequence of values and return the js version of >="
    (is (= "(a)" (ct/eq-gt? :a)))
    (is (= "(true)" (ct/eq-gt?)))
    (is (= "(a >= b)" (ct/eq-gt? :a :b)))
    (is (= "((a >= b) && (b >= c))" (ct/eq-gt? :a :b :c)))))

(deftest eq-lt?
  (testing "should take a sequence of values and return the js version of <="
    (is (= "(a)" (ct/eq-lt? :a)))
    (is (= "(true)" (ct/eq-lt?)))
    (is (= "(a <= b)" (ct/eq-lt? :a :b)))
    (is (= "((a <= b) && (b <= c))" (ct/eq-lt? :a :b :c)))))

(deftest add-directive1*
  (testing "should take an vector - with one element - and a map, returns the vector with the map as the second element"
    (is (= [:el {:b 1}] (ct/add-directive1* [:el] {:b 1})))
    (is (= [:div {:a 1}] (ct/add-directive1* [:div] {:a 1})))))

(deftest add-directive2*
  (testing "should take an vector - with one element - and a map, returns the vector with the map merged with the second element"
    (is (= [:div {:a 1 :b 2}] (ct/add-directive2* [:div {:b 2}] {:a 1})))
    (is (= [:el {:a 1 :b 2 :c 3}] (ct/add-directive2* [:el {:a 1 :c 3}] {:b 2})))))

(deftest add-directive*
  (testing "should take an vector and a map, returns the vector with the map merged with the second element"
    (is (= [:div {:a 1 :b 2} "a"] (ct/add-directive* [:div {:b 2} "a"] {:a 1})))
    (is (= [:el {:a 1 :b 2 :c 3} "b"] (ct/add-directive* [:el {:a 1 :c 3} "b"] {:b 2})))
    (is (= [:el {:a 1 :b 2 :c 3} "b" "c"] (ct/add-directive* [:el {:a 1 :c 3} "b" "c"] {:b 2})))))

(deftest add-directive
  (testing "should take an vector and a map, returns the vector with the map applied as attributes (hiccups template)"
    (is (= [:el {:b 1}] (ct/add-directive [:el] {:b 1})))
    (is (= [:div {:a 1}] (ct/add-directive [:div] {:a 1})))
    (is (= [:div {:a 1 :b 2}] (ct/add-directive [:div {:b 2}] {:a 1})))
    (is (= [:el {:a 1 :b 2 :c 3}] (ct/add-directive [:el {:a 1 :c 3}] {:b 2})))
    (is (= [:div {:a 1 :b 2} "a"] (ct/add-directive [:div {:b 2} "a"] {:a 1})))
    (is (= [:el {:a 1 :b 2 :c 3} "b"] (ct/add-directive [:el {:a 1 :c 3} "b"] {:b 2})))
    (is (= [:el {:a 1 :b 2 :c 3} "b" "c"] (ct/add-directive [:el {:a 1 :c 3} "b" "c"] {:b 2})))))
