(ns clue.template
  (:require [clue.misc :as misc])
  (:refer-clojure :rename {and core-and
                           count core-count
                           for core-for
                           if core-if
                           not core-not
                           or core-or}))

(defn ->directive [directive k]
  (keyword (str directive ":" (misc/->js-str k))))

(defn add-directive1* [body directive]
  [(first body) directive])

(defn add-directive2* [body directive]
  (let [sec (second body)]
    (if (map? sec)
      [(first body) (merge sec directive)]
      [(first body) directive sec])))

(defn add-directive* [body directive]
  (into [(first body) (merge (second body) directive)] (drop 2 body)))

(defn add-directive [body directive]
  (cond
    (= 1 (core-count body)) (add-directive1* body directive)
    (= 2 (core-count body)) (add-directive2* body directive)
    :else (add-directive* body directive)))

(def on (partial ->directive "v-on"))
(def bind (partial ->directive "v-bind"))

(def event "$event")

(defn interpolate [v]
  (misc/->js-str "{{" v "}}"))

(defn for [[el iter] body]
  (let [directive {"v-for" (misc/->js-str el " in " iter)}]
    (add-directive body directive)))

(defn if
  ([cond truthy]
   (let [directive {"v-if" (misc/->js-str cond)}]
     (add-directive truthy directive)))
  ([cond truthy falsy]
   (let [t-directive {"v-if" (misc/->js-str cond)}
         f-directive {"v-else" true}]
     [:template
      (add-directive truthy t-directive)
      (add-directive falsy f-directive)])))

(defn with-params [f & args]
  (let [args-str (->> args
                      (map misc/->js-str)
                      (clojure.string/join ", "))]
    (misc/->js-str f "(" args-str ")")))

(defn op2*
  ([_ a] (misc/->js-str "(" a ")"))
  ([op a b]
   (if b
     (misc/->js-str "(" a " " op " " b ")")
     (op2* op a))))

(defn op*
  ([_] "(true)")
  ([op a] (op2* op a))
  ([op a & vals]
   (reduce (partial op2* op) (into [a] vals))))

(defn expanded-op*
  ([_] "(true)")
  ([op a] (op2* op a))
  ([op a & vals]
   (let [front (butlast (into [a] vals))
         parts (partition 2 (interleave front vals))]
     (->> parts
          (map #(apply op2* op %))
          (reduce (partial op* "&&"))))))

(def and (partial op* "&&"))
(def or (partial op* "||"))
(def eq? (partial expanded-op* "=="))
(def gt? (partial expanded-op* ">"))
(def lt? (partial expanded-op* "<"))
(def eq-gt? (partial expanded-op* ">="))
(def eq-lt? (partial expanded-op* "<="))

(defn count [coll]
  (misc/->js-str "(" coll ".length" ")"))

(defn not
  ([] "(false)")
  ([a] (misc/->js-str "!(" a ")")))
