(ns zilch.virtual-port
  (:require [zilch [mq :as mq]])
  (:import [java.nio ByteBuffer]))

(mq/zeromq-imports)


;; returns kill function
(defn supervise! [thread-fn death-fn]
  (let [supervisor (Thread. (fn []
                              (let [enclosed (Thread. thread-fn)]
                                (.setDaemon enclosed true)
                                (.start enclosed)
                                (try
                                  (loop []
                                    (Thread/sleep 1000)
                                    (if-not (.isAlive enclosed)
                                      (death-fn)
                                      (recur)))
                                (catch InterruptedException e
                                  (.interrupt enclosed)
                                  ))
                                )))]
    (.setDaemon supervisor true)
    (.start supervisor)
    ))

(defn mk-packet [virtual-port ^bytes message]
  (let [bb (ByteBuffer/allocate (+ 2 (count message)))]
    (.putShort bb (short virtual-port))
    (.put bb message)
    (.array bb)
    ))

(defn parse-packet [^bytes packet]
  (let [bb (ByteBuffer/wrap packet)
        port (.getShort bb)
        msg (byte-array (- (count packet) 2))]
    (.get bb msg)
    [port msg]
    ))

(defn virtual-url [port]
  (str "inproc://" port))

(defn- get-virtual-socket! [context mapping-atom port]
  (if-not (contains? @mapping-atom port)
    (swap! mapping-atom
           assoc
           port
           (-> context (mq/socket mq/push) (mq/connect (virtual-url port)))
           ))
  (@mapping-atom port))

(defn close-virtual-sockets! [mapping-atom]
  (doseq [[_ virtual-socket] @mapping-atom]
    (.close virtual-socket))
  (reset! mapping-atom {}))

;; this can't be interrupted, so it's permanent (because msg/recv is a native call)
(defn launch-virtual-port!
  ([context url death-fn]
    (supervise!
      (fn []
        (let [^ZMQ$Socket socket (-> context (mq/socket mq/pull) (mq/bind url))
              virtual-mapping (atom {})]
          (loop []
            (let [[port msg] (parse-packet (mq/recv socket))
                  ^ZMQ$Socket virtual-socket (get-virtual-socket! context virtual-mapping port)]
              ;; TODO: probably need to handle multi-part messages here or something
              (mq/send virtual-socket msg))
            (recur))
          ))
      death-fn))
  ([context url]
    (launch-virtual-port! context url (fn [] (System/exit 1)))
    ))

(defn virtual-send
  ([^ZMQ$Socket socket virtual-port ^bytes message flags]
     (mq/send socket (mk-packet virtual-port message) flags))
  ([^ZMQ$Socket socket virtual-port ^bytes message]
     (virtual-send socket virtual-port message ZMQ/NOBLOCK)))

(defn virtual-bind
  [^ZMQ$Socket socket virtual-port]
  (mq/bind socket (virtual-url virtual-port))
  )
