(ns azlib.sql
  "Basic helpers to work with SQL from clojure"
  (:require [clojure.string :as s])
  (:require [clojure.java.jdbc :as sql])
  (:use clojure.core.memoize))

(def ^:private default-memoize-size 256)

(defonce ^:dynamic *db* nil)
(defonce ^:private ^:dynamic *in-db* false)

(defn- construct-args-list 
  [query]
  (map (comp keyword second) (re-seq #"\W:([\w.-]+)" query)))

(defn- prepare-query 
  [query]
  (s/replace query #"\W:([\w.-]+)" "?"))

(defn setup-db
  [db]
  "Set default db properties."
  (alter-var-root #'*db* merge db))

(defmacro with-db
  [& body]
  "Open new connection to *db* and execute body."
  `(let [b# (fn [] ~@body)]
     (if *in-db* 
       (b#)
       (binding [*in-db* true]
         (sql/with-connection *db* (b#))))))

(defn- check-query-args 
  [al args]
  (when-let [ma (seq (filter (complement (set args)) al))]
    (throw
      (IllegalArgumentException. 
        (str "Unspecified parameters " (vec ma) " in sql.")))))

(defn query* 
  "Execute SQL qeury. Args should be map. 
   Callback should be function of one argument."
  [q args callback]
  (let [al (construct-args-list q)
        pq (prepare-query q)]
    (check-query-args al (keys args))
    (sql/with-query-results*
      (into 
        [pq] 
        (map args al))
      callback)))

(defn one-result 
  "Extract one record from resultset."
  [r]
  (when (< 1 (count r))
    (println (vec r))
    (throw (IllegalStateException. "There are more than 1 records in result table.")))
  (first r))

(defn single-result
  "Extract sinlge value from resultset. Useful for aggreagate functions."
  [r]
  (let [x (one-result r)]
    (when (not= 1 (count r))
      (throw (IllegalStateException. "There are more than 1 columns in result.")))
    (val (first x))))

(defn query-one
  "Query single record."
  ([q] (query-one q {}))
  ([q args] (query* q args one-result)))

(defn query-single 
  "Query single value (aggregate function)."
  ([q] (query-single q {}))
  ([q args] (query* q args single-result)))

(defn query 
  "Execute query. Returns vector."
  ([q] (query q {}))
  ([q args] (query* q args vec)))

(defn- callback-function
  [m]
  (assert 
    (or 
      (not m) 
      (<= 1 (count (filter m [:first :single :one :limit ::func])))))
  (cond
    (::func m) (::func m)
    (:one m) one-result
    (:limit m) (comp vec (partial take (:limit m))) 
    (:first m) first
    (:single m) single-result
    :else vec))

(defn when-args-non-nil 
  [f]
  (fn [& args]
    (when (every? identity args)
      (apply f args))))

(defn- memoize-size [ma]
  (cond
    (number? ma) ma
    (boolean? ma) default-memoize-size
    :else (throw (IllegalArgumentException. (str "Invalid :memoize value " ma)))))

(defmacro defquery
  "Define new query function. 
   Function arguments available as named parameters in sql.
   Example:
     (defquery get-user-by-id [id] \"SELECT * FROM User WHERE id = :id\")
   Additional options can be specefied through meta:
      :one - use 'one-result' as final
      :single - use 'single-result' as final
      :limit - limit result
      :first - use 'first' as callback
      :memoize - memoize results"
  ([name args q] 
    `(defquery ~name ~args ~q nil))
  ([name args q func]
    (assert (vector? args))
    (let [m (meta name)
          m (if func (assoc m ::func func) m)
          callback (callback-function m)]
      `(do
         (defn ~name ~args
           (query* ~q ~(zipmap (map keyword args) args) ~callback))
         ~(when-let [mv (:memoize m)]
            `(alter-var-root (var ~name) #(memo-lru % (memoize-size mv))))
         ~(when (:when-args m)
            `(alter-var-root (var ~name) when-args-non-nil))
         (`~var ~name)))))
