(ns aws.sdb
  (:require [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.sdb.coercion])

  (:import [software.amazon.awssdk.services.simpledb
            SimpleDbClient]

           [software.amazon.awssdk.services.simpledb.model
            Attribute
            CreateDomainRequest
            DeleteAttributesRequest
            DeleteDomainRequest
            DomainMetadataRequest
            DomainMetadataResponse
            GetAttributesRequest
            GetAttributesResponse
            Item
            ListDomainsRequest
            ListDomainsResponse
            PutAttributesRequest
            SelectRequest
            SelectResponse]))

(defn select [^SimpleDbClient client select-request]
  (let [select-fn (fn [req]
                    (->clj (.select client ^SelectRequest (->sdk SelectRequest req))))
        next-batch-fn (fn [{:keys [next-token] :as query_results}]
                        (when next-token
                          (select-fn (assoc select-request :next-token next-token))))]
    (->> (select-fn select-request)
         (iterate next-batch-fn)
         (take-while (complement nil?))
         (mapcat :items))))

(defn put-attributes [^SimpleDbClient client put-attributes-request]
  (->clj (.putAttributes client ^PutAttributesRequest (->sdk PutAttributesRequest put-attributes-request))))

(defn get-attributes [^SimpleDbClient client get-attributes-request]
  (->clj (.getAttributes client ^GetAttributesRequest (->sdk GetAttributesRequest get-attributes-request))))

(defn delete-attributes [^SimpleDbClient client delete-attributes-request]
  (->clj (.deleteAttributes client ^DeleteAttributesRequest (->sdk DeleteAttributesRequest delete-attributes-request))))

(defn list-domains
  ([^SimpleDbClient client]
   (->clj (.listDomains client)))
  ([^SimpleDbClient client list-domains-request]
   (let [list-fn (fn [req]
                   (->clj (.listDomains client ^ListDomainsRequest (->sdk ListDomainsRequest req))))
         next-batch-fn (fn [{:keys [next-token] :as list_results}]
                         (when next-token
                           (list-fn (assoc list-domains-request :next-token next-token))))]
     (->> (list-fn list-domains-request)
          (iterate next-batch-fn)
          (take-while (complement nil?))
          (mapcat :domain-names)))))

(defn create-domain [^SimpleDbClient client domain-name]
  (->clj (.createDomain client ^CreateDomainRequest (->sdk CreateDomainRequest {:domain-name domain-name}))))

(defn delete-domain [^SimpleDbClient client domain-name]
  (->clj (.deleteDomain client ^DeleteDomainRequest (->sdk DeleteDomainRequest {:domain-name domain-name}))))

(defn domain-metadata [^SimpleDbClient client domain-name]
  (->clj (.domainMetadata client ^DomainMetadataRequest (->sdk DomainMetadataRequest {:domain-name domain-name}))))
