(ns aws.s3.client
  (:require [aws.auth :as auth]
            [aws.client]
            [aws.coerce.to-sdk :refer [->sdk]])
  (:import [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.http SdkHttpClient]
           [software.amazon.awssdk.http.apache ApacheHttpClient]
           [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.s3 S3Client S3Configuration]))

(defmethod ->sdk S3Configuration [_ s3-configuration]
  (let [{:keys [accelerate-mode-enabled
                dual-stack-enabled
                path-style-access-enabled]
         :or {accelerate-mode-enabled false
              dual-stack-enabled false
              path-style-access-enabled false}} s3-configuration]
    (.build
     (doto (S3Configuration/builder)
       (.accelerateModeEnabled accelerate-mode-enabled)
       (.dualstackEnabled dual-stack-enabled)
       (.pathStyleAccessEnabled path-style-access-enabled)))))

(defmethod ->sdk S3Client [_ s3-client]
  (let [{:keys [http-client
                service-configuration
                credentials-provider
                endpoint-override
                override-configuration
                region]
         :or {credentials-provider (auth/default-credentials-provider)}} s3-client]
    (.build
     (doto (S3Client/builder)
       (.credentialsProvider credentials-provider)
       (cond-> http-client (.httpClient ^SdkHttpClient (->sdk ApacheHttpClient http-client)))
       (cond-> region (.region (->sdk Region region)))
       (cond-> endpoint-override (.endpointOverride endpoint-override))
       (cond-> service-configuration (.serviceConfiguration ^S3Configuration (->sdk S3Configuration service-configuration)))
       (cond-> override-configuration (.overrideConfiguration ^ClientOverrideConfiguration (->sdk ClientOverrideConfiguration override-configuration)))))))

(defn create
  ([] (S3Client/create))
  ([options]
   (->sdk S3Client options)))
