(ns aws.r53.coercion
  (:require [aws.auth :as auth]
            [aws.client]
            [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.util :as u])
  (:import [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.http SdkHttpClient]
           [software.amazon.awssdk.http.apache ApacheHttpClient]
           [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.route53 Route53Client]
           [software.amazon.awssdk.services.route53.model
            AliasTarget
            Change
            ChangeAction
            ChangeBatch
            ChangeInfo
            ChangeResourceRecordSetsRequest
            ChangeResourceRecordSetsResponse
            GeoLocation
            HostedZone
            HostedZoneConfig
            LinkedService
            ListHostedZonesRequest
            ListHostedZonesResponse
            ListResourceRecordSetsRequest
            ListResourceRecordSetsResponse
            ResourceRecord
            ResourceRecordSet
            ResourceRecordSetFailover
            ResourceRecordSetRegion
            RRType]))

;; ----
;; client
;;
;; ----

(defmethod ->sdk Route53Client [_ route53-client]
  (let [{:keys [http-client
                credentials-provider
                endpoint-override
                override-configuration
                region]
         :or {credentials-provider (auth/default-credentials-provider)}} route53-client]
    (.build
     (doto (Route53Client/builder)
       (.credentialsProvider credentials-provider)
       (cond-> http-client (.httpClient ^SdkHttpClient (->sdk ApacheHttpClient http-client)))
       (cond-> region (.region (->sdk Region region)))
       (cond-> endpoint-override (.endpointOverride endpoint-override))
       (cond-> override-configuration (.overrideConfiguration ^ClientOverrideConfiguration (->sdk ClientOverrideConfiguration override-configuration)))))))

;; ----
;; to clojure
;;
;; ----

(extend-type AliasTarget
  aws.coerce.to-clj/ToClojure
  (to-clj [alias-target]
    (u/only-valid-values
     {:dns-name (.dnsName alias-target)
      :evaluate-target-health (.evaluateTargetHealth alias-target)
      :hosted-zone-id (.hostedZoneId alias-target)})))

(extend-type ChangeInfo
  aws.coerce.to-clj/ToClojure
  (to-clj [change-info]
    (u/only-valid-values
     {:comment (.comment change-info)
      :id (.id change-info)
      :status (.statusAsString change-info)
      :submitted-at (.submittedAt change-info)})))

(extend-type ChangeResourceRecordSetsResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:change-info (->clj (.changeInfo response))})))

(extend-type GeoLocation
  aws.coerce.to-clj/ToClojure
  (to-clj [geo-location]
    (u/only-valid-values
     {:continent-code (.continentCode geo-location)
      :country-code (.countryCode geo-location)
      :subdivision-code (.subdivisionCode geo-location)})))

(extend-type HostedZone
  aws.coerce.to-clj/ToClojure
  (to-clj [hosted-zone]
    (u/only-valid-values
     {:caller-reference (.callerReference hosted-zone)
      :config (->clj (.config  hosted-zone))
      :id (.id hosted-zone)
      :linked-service (when-let [ls (.linkedService hosted-zone)] (->clj ls))
      :name (.name hosted-zone)
      :resource-record-set-count (.resourceRecordSetCount hosted-zone)})))

(extend-type HostedZoneConfig
  aws.coerce.to-clj/ToClojure
  (to-clj [hosted-zone-config]
    (u/only-valid-values
     {:comment (.comment hosted-zone-config)
      :private-zone (.privateZone hosted-zone-config)})))

(extend-type LinkedService
  aws.coerce.to-clj/ToClojure
  (to-clj [linked-service]
    (u/only-valid-values
     {:description (.description linked-service)
      :service-principal (.servicePrincipal linked-service)})))

(extend-type ListHostedZonesResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:hosted-zones (mapv (fn [hosted-zone] (->clj hosted-zone)) (.hostedZones response))
      :is-truncated (.isTruncated response)
      :marker (.marker response)
      :max-items (.maxItems response)
      :next-marker (.nextMarker response)})))

(extend-type ListResourceRecordSetsResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:is-truncated (.isTruncated response)
      :max-items (.maxItems response)
      :next-record-identifier (.nextRecordIdentifier response)
      :next-record-name (.nextRecordName response)
      :next-record-type (.nextRecordTypeAsString response)
      :resource-record-sets (mapv (fn [rr] (->clj rr)) (.resourceRecordSets response))})))

(extend-type ResourceRecord
  aws.coerce.to-clj/ToClojure
  (to-clj [resource-record]
    (u/only-valid-values
     {:value (.value resource-record)})))

(extend-type ResourceRecordSet
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:alias-target (when-let [alias-target (.aliasTarget response)] (->clj alias-target))
      :failover (.failoverAsString response)
      :geo-location (when-let [geo-location (.geoLocation response)] (->clj geo-location))
      :health-check-id (.healthCheckId response)
      :multi-value-answer (.multiValueAnswer response)
      :name (.name response)
      :region (.regionAsString response)
      :resource-records (mapv (fn [rr] (->clj rr)) (.resourceRecords response))
      :set-identifier (.setIdentifier response)
      :traffic-policy-instance-id (.trafficPolicyInstanceId response)
      :ttl (.ttl response)
      :type (.typeAsString response)
      :weight (.weight response)})))

;; ----
;; to sdk
;;
;; ---

(defmethod ->sdk AliasTarget [_ alias-target]
  (let [{:keys [dns-name
                evaluate-target-health
                hosted-zone-id]} alias-target]
    (.build
     (doto (AliasTarget/builder)
       (.dnsName dns-name)
       (.evaluateTargetHealth evaluate-target-health)
       (.hostedZoneId hosted-zone-id)))))

(defmethod ->sdk Change [_ change]
  (let [{:keys [action
                resource-record-set]} change]
    (.build
     (doto (Change/builder)
       (.action (->sdk ChangeAction action))
       (.resourceRecordSet (->sdk ResourceRecordSet resource-record-set))))))

(defn as-change-action-dispatch-fn [t] (class t))
(defmulti ^ChangeAction as-change-action #'as-change-action-dispatch-fn)
(defmethod as-change-action ChangeAction [c] c)
(defmethod as-change-action String [s] (ChangeAction/fromValue s))

(defmethod ->sdk ChangeAction [_ change-action]
  (as-change-action change-action))

(defmethod ->sdk ChangeBatch [_ change-batch]
  (let [{:keys [changes
                comment]} change-batch]
    (.build
     (doto (ChangeBatch/builder)
       (.changes (mapv (fn [change] (->sdk Change change)) changes))
       (cond-> comment (.comment comment))))))

(defmethod ->sdk ChangeResourceRecordSetsRequest [_ change-resource-record-sets-request]
  (let [{:keys [change-batch
                hosted-zone-id]} change-resource-record-sets-request]
    (.build
     (doto (ChangeResourceRecordSetsRequest/builder)
       (.changeBatch  (->sdk ChangeBatch change-batch))
       (.hostedZoneId hosted-zone-id)))))

(defmethod ->sdk GeoLocation [_ geo-location]
  (let [{:keys [continent-code
                country-code
                subdivision-code]} geo-location]
    (.build
     (doto (GeoLocation/builder)
       (cond-> continent-code (.continentCode continent-code)
               country-code (.countryCode country-code)
               subdivision-code (.subdivisionCode subdivision-code))))))

(defmethod ->sdk ListHostedZonesRequest [_ list-hosted-zones-request]
  (let [{:keys [max-items
                marker]
         :or {max-items 100}} list-hosted-zones-request]
    (.build
     (doto (ListHostedZonesRequest/builder)
       (.maxItems (str max-items))
       (cond-> marker (.marker marker))))))

(defmethod ->sdk ListResourceRecordSetsRequest [_ list-resource-record-set-request]
  (let [{:keys [hosted-zone-id
                max-items
                start-record-identifier
                start-record-name
                start-record-type]} list-resource-record-set-request]
    (.build
     (doto (ListResourceRecordSetsRequest/builder)
       (.hostedZoneId hosted-zone-id)
       (cond-> max-items (.maxItems (str max-items))
               start-record-identifier (.startRecordIdentifier start-record-identifier)
               start-record-name (.startRecordName start-record-name)
               start-record-type (.startRecordType (->sdk RRType start-record-type)))))))

(defmethod ->sdk ResourceRecord [_ resource-record]
  (let [{:keys [value]} resource-record]
    (.build
     (doto (ResourceRecord/builder)
       (.value value)))))


(defmethod ->sdk ResourceRecordSet [_ resource-record-set]
  (let [{:keys [alias-target
                failover
                geo-location
                health-check-id
                multi-value-answer
                name
                region
                resource-records
                set-identifier
                traffic-policy-instance-id
                ttl
                type
                weight]} resource-record-set]
    (.build
     (doto (ResourceRecordSet/builder)
       (cond-> alias-target (.alias-target (->sdk AliasTarget alias-target))
               failover (.failover (->sdk ^ResourceRecordSetFailover failover))
               geo-location (.geoLocation (->sdk GeoLocation geo-location))
               health-check-id (.healthCheckId health-check-id)
               (not (nil? multi-value-answer)) (.multiValueAnswer multi-value-answer)
               name (.name name)
               region (.region (->sdk ResourceRecordSetRegion region))
               resource-records (.resourceRecords (mapv (fn [rr] (->sdk ResourceRecord rr)) resource-records))
               set-identifier (.setIdentifier set-identifier)
               traffic-policy-instance-id (.trafficPolicyInstanceId traffic-policy-instance-id)
               ttl (.ttl ttl)
               type (.type (->sdk RRType type))
               weight (.weight weight))))))

(defn as-resource-record-set-failover-dispatch-fn [t] (class t))
(defmulti ^ResourceRecordSetFailover as-resource-record-set-failover #'as-resource-record-set-failover-dispatch-fn)
(defmethod as-resource-record-set-failover ResourceRecordSetFailover [f] f)
(defmethod as-resource-record-set-failover String [s] (ResourceRecordSetFailover/fromValue s))

(defmethod ->sdk ResourceRecordSetFailover [_ failover]
  (as-resource-record-set-failover failover))

(defn as-resource-record-set-region-dispatch-fn [t] (class t))
(defmulti ^ResourceRecordSetRegion  as-resource-record-set-region #'as-resource-record-set-region-dispatch-fn)
(defmethod as-resource-record-set-region ResourceRecordSetRegion [r] r)
(defmethod as-resource-record-set-region String [s] (ResourceRecordSetRegion/fromValue s))

(defmethod ->sdk ResourceRecordSetRegion [_ region]
  (as-resource-record-set-region region))

(defn as-resource-record-type-dispatch-fn [t] (class t))
(defmulti ^RRType as-resource-record-type #'as-resource-record-type-dispatch-fn)
(defmethod as-resource-record-type RRType [t] t)
(defmethod as-resource-record-type String [s] (RRType/fromValue s))

(defmethod ->sdk RRType [_ resource-record-type]
  (as-resource-record-type resource-record-type))
