(ns aws.r53
  (:require [aws.auth :as auth]
            [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.r53.coercion])
  (:import [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.route53 Route53Client]
           [software.amazon.awssdk.services.route53.model
            ChangeResourceRecordSetsRequest
            ListResourceRecordSetsRequest
            ListHostedZonesRequest]))

(defn ^Route53Client client
  ([] (client {:region Region/AWS_GLOBAL}))
  ([{:keys [region] :or {region Region/AWS_GLOBAL} :as options}]
   (->sdk Route53Client (merge options {:region region}))))

(defn list-hosted-zones
  ([client] (list-hosted-zones client {}))
  ([^Route53Client client request]
   (let [list-fn (fn [req]
                   (->clj
                    (.listHostedZones
                     client
                     ^ListHostedZonesRequest
                     (->sdk ListHostedZonesRequest req))))

         next-batch-fn (fn [{:keys [next-marker] :as response}]
                         (when next-marker
                           (list-fn (assoc request :marker next-marker))))]

     (->>  (list-fn request)
           (iterate next-batch-fn)
           (take-while (complement nil?))
           (mapcat :hosted-zones)))))

(defn list-resource-record-sets
  [^Route53Client client request]
  (let [list-fn (fn [req]
                  (->clj
                   (.listResourceRecordSets
                    client
                    ^ListResourceRecordSetsRequest
                    (->sdk ListResourceRecordSetsRequest req))))

        next-batch-fn (fn [{:keys [is-truncated next-record-name next-record-type] :as response}]
                        (when is-truncated
                          (list-fn (assoc request
                                          :start-record-name next-record-name
                                          :start-record-type next-record-type))))]

    (->>  (list-fn request)
          (iterate next-batch-fn)
          (take-while (complement nil?))
          (mapcat :resource-record-sets))))

(defn change-resource-record-sets [client change-resource-record-sets-request]
  (->clj (.changeResourceRecordSets client ^ChangeResourceRecordSetsRequest (->sdk ChangeResourceRecordSetsRequest change-resource-record-sets-request))))
