(ns aws.client
  (:require [aws.coerce.to-sdk :refer [->sdk]])
  (:import [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.core.retry RetryPolicy]
           [software.amazon.awssdk.core.retry.backoff
            BackoffStrategy
            EqualJitterBackoffStrategy
            FixedDelayBackoffStrategy
            FullJitterBackoffStrategy]
           [software.amazon.awssdk.core.retry.conditions
            AndRetryCondition
            MaxNumberOfRetriesCondition
            OrRetryCondition
            RetryCondition
            RetryOnExceptionsCondition
            RetryOnStatusCodeCondition]
           [software.amazon.awssdk.http.apache ApacheHttpClient ProxyConfiguration]))


(defmethod ->sdk ApacheHttpClient [_ apache-http-client]
  (let [{:keys [connection-max-idle-time
                connection-timeout
                connection-time-to-live
                expect-continue-enabled
                local-address
                max-connections
                proxy-configuration
                socket-timeout]} apache-http-client]
    (.build
     (doto (ApacheHttpClient/builder)
       (cond-> connection-max-idle-time (.connectionMaxIdleTime connection-max-idle-time)
               connection-timeout (.connectionTimeout connection-timeout)
               connection-time-to-live (.connectionTimeToLive connection-time-to-live)
               expect-continue-enabled (.expectContinueEnabled expect-continue-enabled)
               local-address (.localAddress local-address)
               max-connections (.maxConnections (int max-connections))
               proxy-configuration (.proxyConfiguration ^ProxyConfiguration (->sdk ProxyConfiguration proxy-configuration))
               socket-timeout (.socketTimeout socket-timeout))))))

(defmethod ->sdk ClientOverrideConfiguration [_ client-override-configuration]
  (let [{:keys [advanced-options
                headers
                retry-policy]} client-override-configuration]
    (.build
     (doto (ClientOverrideConfiguration/builder)
       (cond-> advanced-options (.advancedOptions advanced-options)
               headers (.headers headers)
               retry-policy(.retryPolicy (->sdk retry-policy)))))))

(defmethod ->sdk ProxyConfiguration [_ proxy-configuration]
  (let [{:keys [endpoint
                non-proxy-hosts
                password
                preemptive-basic-authentication-enabled
                username]} proxy-configuration]
    (.build
     (doto (ProxyConfiguration/builder)
       (cond-> endpoint (.endpoint endpoint)
               non-proxy-hosts (.nonProxyHosts non-proxy-hosts)
               password (.password password)
               preemptive-basic-authentication-enabled (.preemptiveBasicAuthenticationEnabled preemptive-basic-authentication-enabled)
               username (.username username))))))

(defmethod ->sdk RetryPolicy [_ retry-policy]
  (let [{:keys [backoff-strategy
                num-retries
                retry-condition]} retry-policy]
    (.build
     (doto (RetryPolicy/builder)
       (cond-> backoff-strategy (.backoffStrategy backoff-strategy)
               num-retries (.numRetries num-retries)
               retry-condition (.retryCondition retry-condition))))))

(defmethod ->sdk EqualJitterBackoffStrategy [_ equal-jitter-backoff-strategy]
  (let [{:keys [base-delay
                max-backoff-time]} equal-jitter-backoff-strategy]
    (.build
     (doto (EqualJitterBackoffStrategy/builder)
       (.baseDelay base-delay)
       (.maxBackoffTime max-backoff-time)))))

(defmethod ->sdk FixedDelayBackoffStrategy [_ fixed-delay-backoff-strategy]
  (let [{:keys [fixed-backoff]} fixed-delay-backoff-strategy]
    (FixedDelayBackoffStrategy/create fixed-backoff)))

(defmethod ->sdk FullJitterBackoffStrategy [_ full-jitter-backoff-strategy]
  (let [{:keys [base-delay
                max-backoff-time]} full-jitter-backoff-strategy]
    (.build
     (doto (FullJitterBackoffStrategy/builder)
       (.baseDelay base-delay)
       (.maxBackoffTime max-backoff-time)))))

(defn no-retry-policy []
  (RetryPolicy/none))

(defn default-retry-policy []
  (RetryPolicy/defaultRetryPolicy))

(defn default-backoff-strategy []
  (BackoffStrategy/defaultStrategy))

(defn default-retry-condition []
  (RetryCondition/defaultRetryCondition))

(defn  and-retry-condition [retry-conditions]
  (AndRetryCondition/create (into-array RetryCondition retry-conditions)))

(defn max-number-of-retries-condition [max-number-of-retries]
  (MaxNumberOfRetriesCondition/create max-number-of-retries))

(defn or-retry-condition [retry-conditions]
  (OrRetryCondition/create (into-array RetryCondition retry-conditions)))

(defn retry-on-exceptions-condition [exceptions]
  (RetryOnExceptionsCondition/create (into-array java.lang.Class exceptions)))

(defn retry-on-status-code-condition [status-codes]
  (RetryOnStatusCodeCondition/create (into-array Integer (mapv (fn [i] (int i)) status-codes))))
