(ns awohletz.validations)

(defrecord ValidationResult [status reason])

(defprotocol ValidateableFn
  (valid? [_ args success error] "Runs the validations and calls success or error depending on the result.
  success will be passed the actual validated function with its args, ready to run.
  error will be passed all the ValidationResults that had errors"))

;Core functions ----------------------------------------------------------------------

(defn combine-validations
  "Combine a seq of validations into one validation function that will run them all in the order
  they appear in the seq. Returns a ValidationResult of the first validation that failed.
  If all validations succeed, returns nil"
  ([validations]
   (combine-validations validations []))
  ([validations prefix]
   (fn [args]
     (filter #(= (:status %) :fail)
             (map
               (fn [validation]
                 (validation (get-in args prefix)))
               validations)))))

(defn with-validations
  "Returns a new ValidateableFn that wraps a function f.
  Assumes the wrapped function f takes a single hashmap as a parameter.
  prefix = a sequence of keys to access a nested hashmap in the args to run the validations on"
  {:arglists '([f & validations] [f prefix & validations])}
  [f & validations]
  (let [prefix (if (vector? (first validations)) (first validations) [])
        validator (combine-validations
                    (if (vector? (first validations)) (rest validations) validations)
                    prefix)]
    (reify ValidateableFn
      (valid? [_ args success error]
        (let [errors (validator args)]
          (if (> (count errors) 0)
            (error errors)
            (success #(f args))))))))

;Utils ----------------------------------------------------------------------

(defn dissoc-in
  [m [k & ks]]
  (if ks
    (assoc m k (dissoc-in (get m k) ks))
    (dissoc m k)))

(defn contains-only-empty-maps?
  [m]
  (every?
    (fn [[k v]]
      (if (map? v)
        (if (== 0 (count v))
          true
          (contains-only-empty-maps? v))
        false))
    m))

(defn ensure-int
  "Will try to parse an int if it is a string. If it's unparseable, will return null"
  [n]
  (if-not (integer? n)
    (try
      (Integer/parseInt (str n))
      (catch NumberFormatException e))
    n))

(defn ensure-num
  "Determine if a var contains a number that could parsed with Float/parseFloat"
  [n]
  (if-not (number? n)
    (try
      (Float/parseFloat n)
      (catch NumberFormatException e))
    n))

(defn ensure-vec
  "Coerce the given into a vector if it isn't already"
  [x]
  (vec (flatten (merge [] x))))

;Wrappers ----------------------------------------------------------------------

(defn wrap-presence
  "Validate that the parameter is present in the map. If it is, will return the return value of vf

  korks = keyword or vector of keywords that we access the parameter to validate with. If a param under
  the keywords can't be found, will convert keywords to strings and look again

  vf = function that will receive the args if it is present. Will add the present value to args under
  key ::value"
  [vf korks]
  (let [korks (ensure-vec korks)]
    (fn [args]
      (if-let [val (get-in args korks (get-in args (mapv name korks)))]
        (vf (assoc args ::value val))
        (ValidationResult. :fail (str korks " is not present."))))))

(defn wrap-format
  "Wrap in format validation"
  [vf korks pattern]
  (wrap-presence
    (fn [{:keys [::value] :as args}]
      (if (re-matches pattern value)
        (vf args)
        (ValidationResult. :fail (str korks " value " value " does not match pattern " pattern))))
    korks))

(defn wrap-numericality
  "Wrap in numericality validation"
  [vf korks {:keys [only-integer] :as opts}]
  (wrap-presence
    (fn [{:keys [::value] :as args}]
      (let [fail (ValidationResult. :fail (str korks " value " value " is not numeric with options: " opts))]
        (if only-integer
          (if (ensure-int value)
            (vf args)
            fail)
          (if (ensure-num value)
            (vf args)
            fail))))
    korks))

;Validators ----------------------------------------------------------------------

(def success (fn [_] (ValidationResult. :success "All validations passed")))

(defn presence
  "Validate that the parameter is present in the map. If it is, will return the return value of vf

  korks = keyword or vector of keywords that we access the parameter to validate with. If a param under
  the keywords can't be found, will convert keywords to strings and look again"
  [korks]
  (wrap-presence success korks))

(defn pattern
  "Validate that a param value matches a regex"
  [korks pattern]
  (wrap-format success korks pattern))

(defn numericality
  "Validate that a parameter is a number. Options:

  :only-integer = true/false -- Specifies that only integer values are allowed. Otherwise
  matches all numbers. Defaults to false."
  ([korks]
   (wrap-numericality success korks {}))
  ([korks opts]
   (wrap-numericality success korks opts)))

(defn dissoc-all [m korks-coll]
  (reduce (fn [new-m korks]
            (dissoc-in new-m (ensure-vec korks)))
          m
          korks-coll))

(defn whitelist
  "Validate that only the given parameters are present."
  [korks-coll]
  (fn [args]
    (let [disallowed-keys (dissoc-all args korks-coll)]
      (if (contains-only-empty-maps? disallowed-keys)
        success
        (ValidationResult. :fail (str "Keys not whitelisted: " disallowed-keys))))))