(ns aviary.serve
  (:require [aviary.system :as system]
            [aviary.console :as console]
            [bidi.ring :as bidi]
            [bidi.bidi :refer [compile-route]]
            [ring.util.response :as ring]
            [org.httpkit.server :refer [run-server]]))

(defn- ensure-trailing-slash
  "Use regular expression (with negative lookbehind) to ensure a trailing slash"
  [path]
  (.replaceAll path "(?<!/)$" "/"))

(defn- make-resource-routes [resources]
  (when resources
    (map
      (fn [path]
        ["/" (bidi/resources-maybe {:prefix
                                   (ensure-trailing-slash path)})])
      resources)))

(defn- make-manifest-handler [content-type content]
  (fn [req]
    (-> (content)
        (ring/response)
        (ring/content-type content-type))))

(defn- make-manifest-routes [manifests]
  (when manifests
    (mapv
      (fn [[content-type manifest]]
        ["/" (mapv
               (fn [[path content]]
                 [path (make-manifest-handler content-type content)])
               (manifest))])
      manifests)))

(defn- make-logging-handler [routes]
  (fn [req]
    (let [res ((bidi/make-handler routes) req)
          key (keyword "http" (name (:request-method req)))
          path (:uri req)
          code (or (:status res) 404)]
      (cond
        (<= code 299) (console/info key (format "[%s] %s" code path))
        (<= code 399) (console/warn key (format "[%s] %s" code path))
        (<= code 599) (console/error key (format "[%s] %s" code path)))
      res)))

(system/defcomponent serve [config]
  ([_] (let [routes ["" (vec
                          (concat
                            (-> config :resources make-resource-routes)
                            (-> config :manifests make-manifest-routes)))]
             handler (if-not (:static? config)
                       (make-logging-handler routes)
                       (bidi/make-handler
                         (compile-route routes)))]
         (run-server
           handler (select-keys config [:port]))))
  ([_ stop] (stop)))
