(ns duct.component.hikaricp
  (:require [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]
            [clojure.spec :as s]
            [dire.core :refer [with-handler! with-finally!]]
            )
  (:import [com.zaxxer.hikari HikariConfig HikariDataSource]))

(s/def  ::hikaricfg (s/keys* :req [::uri]
                             :opt [::username
                                  ::password
                                  ::auto-commit
                                  ::conn-timeout
                                  ::idle-timeout
                                  ::max-lifetime
                                  ::max-pool-size
                                  ::min-idle
                                  ::pool-name]
                            )
  )

(def jdbc-regex #"jdbc:[a-zA-Z0-9._+-]+:[a-zA-Z0-9._/;=:]+")

(s/def ::jdbc-type     (s/and string? #(re-matches jdbc-regex %)))
(s/def ::stringornil   (s/nilable string?))
(s/def ::jdbc-timeout  (s/nilable (s/and pos-int? #(> % 250))))
(s/def ::posintornil   (s/nilable pos-int?))

(defn- make-config
  [{:keys [uri username password auto-commit? conn-timeout idle-timeout
           max-lifetime conn-test-query min-idle max-pool-size pool-name]}]
  {:pre  [(s/valid? ::jdbc-type     uri)
          (s/valid? ::stringornil   username)
          (s/valid? ::stringornil   password)
          (s/valid? ::jdbc-timeout  conn-timeout)
          (s/valid? ::jdbc-timeout  idle-timeout)
          (s/valid? ::posintornil   max-lifetime)
          (s/valid? ::posintornil   max-pool-size)
          (s/valid? ::jdbc-timeout  min-idle)
          (s/valid? ::stringornil   pool-name)
          ]
   }
  (let [cfg (HikariConfig.)]
    (when uri                  (.setJdbcUrl cfg uri))
    (when username             (.setUsername cfg username))
    (when password             (.setPassword cfg password))
    (when (some? auto-commit?) (.setAutoCommit cfg auto-commit?))
    (when conn-timeout         (.setConnectionTimeout cfg conn-timeout))
    (when idle-timeout         (.setIdleTimeout cfg conn-timeout))
    (when max-lifetime         (.setMaxLifetime cfg max-lifetime))
    (when max-pool-size        (.setMaximumPoolSize cfg max-pool-size))
    (when min-idle             (.setMinimumIdle cfg min-idle))
    (when pool-name            (.setPoolName cfg pool-name))
    cfg))

(defn- make-spec [component]
  {:datasource (HikariDataSource. (make-config component))})

(defrecord HikariCP [uri]
  component/Lifecycle
  (start [component]
    (if-let [spec (:spec component)]
      (do
        (log/debug "HikariCP component already started.")
        component)
      (do
        (log/debug "Starting HikariCP component")
        (assoc component :spec (make-spec component)))
      )
    )
  (stop [component]
    (if-let [spec (:spec component)]
      (do (log/debug "Stopping HikariCP component")
          (.close (:datasource spec))
          (dissoc component :spec))
      component)
    )
  )

(defn hikaricp [options]
  "Make sure at least an URI is in the options"
  {:pre [(:uri options)]}
  (map->HikariCP options))
