(ns reagent-material-ui.scripts
  (:require [clojure.edn :as edn]
            [clojure.string :as str]
            [camel-snake-kebab.core :as csk]
            [clojure.java.io :as io]
            [instaparse.core :as insta])
  (:import (java.io File)))

(def exclude-clj #{"comment" "compare" "filter" "list" "loop" "map" "print" "remove" "repeat" "shuffle" "sort" "update"})
(def js-keyword-icons #{"class" "delete" "public"})
(def js-keyword-components #{"switch"})

(defn component-ns-name [js-name]
  (let [clj-name (csk/->kebab-case js-name)]
    (if (contains? js-keyword-components clj-name)
      (str clj-name "-component")
      clj-name)))

(defn icon-ns-name [js-name]
  (let [clj-name (csk/->kebab-case js-name)]
    (if (contains? js-keyword-icons clj-name)
      (str clj-name "-icon")
      clj-name)))

(defn generate-core-ns-content [js-name]
  (let [clj-name (csk/->kebab-case js-name)]
    (str "(ns reagent-material-ui.core." (component-ns-name js-name) \newline
         "  \"Imports @material-ui/core/" js-name " as a Reagent component." \newline
         "   Original documentation is at https://material-ui.com/api/" clj-name "/ .\"" \newline
         (when (contains? exclude-clj clj-name)
           (str "  (:refer-clojure :exclude [" clj-name "])" \newline))
         "  (:require [reagent-material-ui.util :refer [adapt-react-class]]" \newline
         "            [material-ui]))" \newline \newline
         "(def " clj-name " (adapt-react-class (.-" js-name " js/MaterialUI) \"mui-" clj-name "\"))" \newline)))

(defn write-core-ns [js-name]
  (let [content (generate-core-ns-content js-name)
        path (str "./src/core/reagent_material_ui/core/" (csk/->snake_case (component-ns-name js-name)) ".cljs")]
    (spit path content)))

(defn generate-common-core-ns-content [js-names]
  (let [clj-names (map csk/->kebab-case js-names)]
    (str "(ns reagent-material-ui.components" \newline
         "  \"Imports all components from @material-ui/core as Reagent components." \newline
         "   Original documentation is at https://material-ui.com/api/ .\"" \newline
         "  (:refer-clojure :exclude [" (str/join " " (keep exclude-clj clj-names)) "])" \newline
         "  (:require "
         (str/join "\n            " (for [clj-name clj-names]
                                      (str "reagent-material-ui.core." (component-ns-name clj-name))))
         "))" \newline \newline
         (str/join \newline (for [clj-name clj-names]
                              (str "(def " clj-name " reagent-material-ui.core." (component-ns-name clj-name) "/" clj-name ")")))
         \newline)))

(defn write-core [names-file]
  (let [js-names (edn/read-string (slurp names-file))]
    (doseq [js-name js-names]
      (write-core-ns js-name))
    (spit "./src/core/reagent_material_ui/components.cljs" (generate-common-core-ns-content js-names))))

(def ebnf "
S = element <','>?
<element> = empty-element | nonempty-element
nonempty-element = <'<'> tag proplist <'>'> element* <'</'> <tag> <'>'>
empty-element = <'<'> tag proplist <'/>'>
proplist = propvalue*
<propvalue> = prop <'=\"'> value <'\"'>
<value> = #'[^\"]*'
<prop> = 'clipRule'|'cx'|'cy'|'d'|'fill'|'fillOpacity'|'fillRule'|'id'|'opacity'|'r'|'transform'
<tag> = 'React.Fragment'|'circle'|'defs'|'g'|'path'
")

(def parser (insta/parser ebnf :auto-whitespace (insta/parser "whitespace = #'\\s+'")))

(defn element->react [element]
  (let [[type tag proplist & children] element
        react-tag (if (= "React.Fragment" tag)
                    '(.-Fragment js/React)
                    tag)
        props (some->> (next proplist)
                       (apply hash-map))
        parsed-children (for [child children]
                          (if (vector? child)
                            (element->react child)
                            (throw (ex-info "Illegal child" {:element element
                                                             :child   child}))))]
    `(~'e ~react-tag ~@(when props [(symbol "#js")]) ~props ~@parsed-children)))

(defn read-icon [path]
  (with-open [rdr (io/reader path)]
    (let [jsx (nth (line-seq rdr) 4)
          parsed (parser jsx)]
      (if (insta/failure? parsed)
        (throw (ex-info "Failed to parse" {:path   path
                                           :jsx    jsx
                                           :result parsed}))
        (element->react (second parsed))))))

(defn generate-icon-ns-content [js-name content]
  (let [clj-name (csk/->kebab-case js-name)]
    (str "(ns reagent-material-ui.icons." (icon-ns-name js-name) \newline
         "  \"Imports @material-ui/icons/" js-name " as a Reagent component.\"" \newline
         (when (contains? exclude-clj clj-name)
           (str "  (:refer-clojure :exclude [" clj-name "])" \newline))
         "  (:require-macros [reagent-material-ui.macro :refer [e]])" \newline
         "  (:require [reagent-material-ui.util :refer [create-svg-icon]]))" \newline \newline
         "(def " clj-name " (create-svg-icon " (pr-str content) \newline
         (str/join (repeat (+ 23 (count clj-name)) \space))
         \" js-name "\"))" \newline)))

(defn write-icon-ns [js-name icon-path]
  (let [icon-content (read-icon icon-path)
        clj-path (str "./src/icons/reagent_material_ui/icons/" (csk/->snake_case (icon-ns-name js-name)) ".cljs")
        ns-content (generate-icon-ns-content js-name icon-content)]
    (spit clj-path ns-content)))

(defn write-icons [^String mui-path]
  (let [f (File. mui-path)
        filenames (seq (.list f))
        js-names (sort (for [filename filenames
                             :let [[_ js-name :as matches?] (re-matches #"([A-Z].*)\.js" filename)]
                             :when matches?]
                         js-name))]
    (doseq [js-name js-names]
      (write-icon-ns js-name (str mui-path "/" js-name ".js")))))

(defn color->string [[js-name values]]
  (let [clj-name (csk/->kebab-case js-name)
        indent (str/join (repeat (+ 7 (count clj-name)) \space))]
    (str "(def " clj-name " {"
         (str/join (str \newline indent)
                   (for [[key value] values
                         :let [num-key? (contains? #{\0 \1 \2 \3 \4 \5 \6 \7 \8 \9} (first key))]]
                     (str (when-not num-key?
                            \:)
                          key
                          (when num-key? (str/join (repeat (- 5 (count key)) \space)))
                          \space \" value \")))
         "})" \newline)))

(defn generate-color-ns-content [colors]
  (str "(ns reagent-material-ui.colors" \newline
       "  \"Imports all colors from @material-ui/core/colors." \newline
       "   Original documentation is at https://material-ui.com/customization/color/ .\")" \newline \newline
       (str/join \newline (map color->string colors))))

(defn read-color-js-file [^File f]
  (with-open [rdr (io/reader f)]
    (doall
     (for [line (line-seq rdr)
           :let [[_ key color :as matches?] (re-matches #" *([0-9A]*|black|white): '([#0-9a-f]*)'," line)]
           :when matches?]
       [key color]))))

(defn write-colors [^String mui-path]
  (let [index (File. mui-path "index.js")
        js-names (with-open [rdr (io/reader index)]
                   (doall
                    (for [line (line-seq rdr)]
                      (let [[_ color] (re-matches #".*\{ default as (.*) \}.*" line)]
                        color))))
        colors (for [js-name js-names]
                 [js-name (read-color-js-file (File. mui-path (str js-name ".js")))])]
    (spit "./src/core/reagent_material_ui/colors.cljs" (generate-color-ns-content colors))))

