(ns ^{:doc "Namespace to be executed inside the boot-dynamodb pod"}
    boot-dynamodb.pod
  (:require [clojure.string :as str]
            [boot.util :as util]
            [camel-snake-kebab.core :as csk]
            [boot.pod :as pod])
  (:import (com.amazonaws.services.dynamodbv2.local main.ServerRunner
                                                    server.DynamoDBProxyServer)))

(def ^:private dynamo-option-set #{:cors :db-path :delay-transient-statuses
                                   :in-memory :optimize-db-before-startup
                                   :port :shared-db :dynamodb-help})

(def ^:private java-option-set #{:log4j-path})

(defn dynamodb-opt-strs
  "For each key-value in input, return one or more option strings (in a
  vector)."
  [[k v]]
  (cond
    (= :dynamodb-help k) ["-help"]
    (instance? Boolean v) (if v [(str "-" (csk/->camelCaseString k))] []) ;; no boolean? in 1.8
    (keyword? v) [(str "-" (csk/->camelCaseString k)) (name v)]
    :else [(str "-" (csk/->camelCaseString k)) (str v)]))

(def ^:private dynamodb-option-xf
  (comp (filter (comp dynamo-option-set first))
        (mapcat dynamodb-opt-strs)))

(defn java-prop-str
  [[k v]]
  {:pre [(string? k) (string? v)]}
  (str "-D" k "=" v))

(defn dynamodb-cmd-line-opts [opts]
  (into [] dynamodb-option-xf opts))

(def ^:private java-prop-xf
  (comp (filter (comp string? first))
        (map java-prop-str)))

(defn java-cmd-line-opts [opts]
  (into [] java-prop-xf opts))

(defonce ^:private server (atom nil))

(defn- set-sqlite-native! [sqlite-native-dep]
  (log/warn "%s\n" (:repositories pod/env))
  (let [sqlite-resolved-path (->> sqlite-native-dep
                                  (pod/resolve-dependency-jar pod/env)
                                  str)]
    (util/dbug* "Sqlite4java resolved to %s\n" sqlite-resolved-path)
    (System/setProperty "java.library.path" sqlite-resolved-path)))

(defn start! ^DynamoDBProxyServer [opts]
  (let [args (dynamodb-cmd-line-opts opts)
        sqlite-native-dep (:sqlite-native-dep opts)
        help? (some #{"-help"} args)]
    (try
      (util/dbug* "Setting up Sqlite4java Native Library %s\n" sqlite-native-dep)
      (set-sqlite-native! sqlite-native-dep)
      (util/dbug* "Starting server with args %s\n" (util/pp-str args))
      (reset! server
              (let [server (ServerRunner/createServerFromCommandLineArgs (into-array String args))]
                (when-not help?
                  (doto server (.start)))))
      (catch Exception e
        (if-not help?
          (util/print-ex e)
          (util/dbug* "Swooshed exception: %s\n" (.getMessage e)))))))

(defn stop! []
  (when-let [s ^DynamoDBProxyServer @server]
    (try
      (.stop s)
      (catch Exception e
        (util/print-ex e))
      (finally
        (try
          (.stop server)
          (catch Exception e
            (util/dbug* "Swooshed exception: %s\n" (.getMessage e))))))))
