/*
 * Decompiled with CFR 0.152.
 */
package com.google.appengine.repackaged.com.google.common.net;

import com.google.appengine.repackaged.com.google.common.base.Hash;
import com.google.appengine.repackaged.com.google.common.base.Preconditions;
import com.google.appengine.repackaged.com.google.common.base.Splitter;
import com.google.appengine.repackaged.com.google.common.io.ByteStreams;
import com.google.appengine.repackaged.com.google.common.net.InetAddresses;
import com.google.appengine.repackaged.com.google.common.primitives.Ints;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import sun.net.util.IPAddressUtil;

public final class InetAddresses {
    private static final Splitter SPLITTER = Splitter.on('.');
    private static final Inet4Address LOOPBACK4 = (Inet4Address)InetAddresses.forString("127.0.0.1");
    private static final Inet4Address ANY4 = (Inet4Address)InetAddresses.forString("0.0.0.0");

    private InetAddresses() {
    }

    private static Inet4Address getInet4Address(byte[] bytes) {
        Preconditions.checkArgument(bytes.length == 4, "Byte array has invalid length for an IPv4 address: %s != 4.", bytes.length);
        try {
            InetAddress ipv4 = InetAddress.getByAddress(bytes);
            if (!(ipv4 instanceof Inet4Address)) {
                throw new UnknownHostException(String.format("'%s' is not an IPv4 address.", ipv4.getHostAddress()));
            }
            return (Inet4Address)ipv4;
        }
        catch (UnknownHostException e) {
            throw new IllegalArgumentException(String.format("Host address '%s' is not a valid IPv4 address.", Arrays.toString(bytes)), e);
        }
    }

    private static boolean sanityCheckIPv4String(String ipv4String) {
        int count = 0;
        for (String part : SPLITTER.split(ipv4String)) {
            if (++count > 4) {
                return false;
            }
            try {
                if (part.equals(Integer.toString(Integer.parseInt(part)))) continue;
                return false;
            }
            catch (NumberFormatException nfe) {
                return false;
            }
        }
        return count == 4;
    }

    public static InetAddress forString(String ipString) {
        byte[] addr = IPAddressUtil.textToNumericFormatV4(ipString);
        if (addr != null) {
            if (!InetAddresses.sanityCheckIPv4String(ipString)) {
                addr = null;
            }
        } else {
            addr = IPAddressUtil.textToNumericFormatV6(ipString);
        }
        if (addr == null) {
            throw new IllegalArgumentException(String.format("'%s' is not an IP string literal.", ipString));
        }
        try {
            return InetAddress.getByAddress(addr);
        }
        catch (UnknownHostException e) {
            throw new IllegalArgumentException(String.format("'%s' is extremely broken.", ipString), e);
        }
    }

    public static boolean isInetAddress(String ipString) {
        try {
            InetAddresses.forString(ipString);
            return true;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static String toUriString(InetAddress ip) {
        if (ip instanceof Inet6Address) {
            return "[" + ip.getHostAddress() + "]";
        }
        return ip.getHostAddress();
    }

    public static InetAddress forUriString(String hostAddr) {
        Preconditions.checkNotNull(hostAddr);
        Preconditions.checkArgument(!hostAddr.isEmpty(), "host string is empty");
        InetAddress retval = null;
        try {
            retval = InetAddresses.forString(hostAddr);
            if (retval instanceof Inet4Address) {
                return retval;
            }
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        if (!hostAddr.startsWith("[") || !hostAddr.endsWith("]")) {
            throw new IllegalArgumentException("Not a valid address: \"" + hostAddr + '\"');
        }
        retval = InetAddresses.forString(hostAddr.substring(1, hostAddr.length() - 1));
        if (retval instanceof Inet6Address) {
            return retval;
        }
        throw new IllegalArgumentException("Not a valid address: \"" + hostAddr + '\"');
    }

    public static boolean isUriInetAddress(String ipString) {
        try {
            InetAddresses.forUriString(ipString);
            return true;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isCompatIPv4Address(Inet6Address ip) {
        if (!ip.isIPv4CompatibleAddress()) {
            return false;
        }
        byte[] bytes = ip.getAddress();
        return bytes[12] != 0 || bytes[13] != 0 || bytes[14] != 0 || bytes[15] != 0 && bytes[15] != 1;
    }

    public static Inet4Address getCompatIPv4Address(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.isCompatIPv4Address(ip), "Address '%s' is not IPv4-compatible.", ip.getHostAddress());
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip.getAddress(), 12, 16));
    }

    public static boolean is6to4Address(Inet6Address ip) {
        byte[] bytes = ip.getAddress();
        return bytes[0] == 32 && bytes[1] == 2;
    }

    public static Inet4Address get6to4IPv4Address(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.is6to4Address(ip), "Address '%s' is not a 6to4 address.", ip.getHostAddress());
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip.getAddress(), 2, 6));
    }

    public static boolean isTeredoAddress(Inet6Address ip) {
        byte[] bytes = ip.getAddress();
        return bytes[0] == 32 && bytes[1] == 1 && bytes[2] == 0 && bytes[3] == 0;
    }

    public static TeredoInfo getTeredoInfo(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.isTeredoAddress(ip), "Address '%s' is not a Teredo address.", ip.getHostAddress());
        byte[] bytes = ip.getAddress();
        Inet4Address server = InetAddresses.getInet4Address(Arrays.copyOfRange(bytes, 4, 8));
        int flags = ByteStreams.newDataInput(bytes, 8).readShort() & 0xFFFF;
        int port = ~ByteStreams.newDataInput(bytes, 10).readShort() & 0xFFFF;
        byte[] clientBytes = Arrays.copyOfRange(bytes, 12, 16);
        for (int i = 0; i < clientBytes.length; ++i) {
            clientBytes[i] = ~clientBytes[i];
        }
        Inet4Address client = InetAddresses.getInet4Address(clientBytes);
        return new TeredoInfo(server, client, port, flags);
    }

    public static boolean isIsatapAddress(Inet6Address ip) {
        if (InetAddresses.isTeredoAddress(ip)) {
            return false;
        }
        byte[] bytes = ip.getAddress();
        if ((bytes[8] | 3) != 3) {
            return false;
        }
        return bytes[9] == 0 && bytes[10] == 94 && bytes[11] == -2;
    }

    public static Inet4Address getIsatapIPv4Address(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.isIsatapAddress(ip), "Address '%s' is not an ISATAP address.", ip.getHostAddress());
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip.getAddress(), 12, 16));
    }

    public static Inet4Address getEmbeddedIPv4ClientAddress(Inet6Address ip) {
        if (InetAddresses.isCompatIPv4Address(ip)) {
            return InetAddresses.getCompatIPv4Address(ip);
        }
        if (InetAddresses.is6to4Address(ip)) {
            return InetAddresses.get6to4IPv4Address(ip);
        }
        if (InetAddresses.isTeredoAddress(ip)) {
            return InetAddresses.getTeredoInfo(ip).getClient();
        }
        throw new IllegalArgumentException(String.format("'%s' has no embedded IPv4 address.", ip.getHostAddress()));
    }

    public static Inet4Address getCoercedIPv4Address(InetAddress ip) {
        if (ip instanceof Inet4Address) {
            return (Inet4Address)ip;
        }
        byte[] bytes = ip.getAddress();
        boolean leadingBytesOfZero = true;
        for (int i = 0; i < 15; ++i) {
            if (bytes[i] == 0) continue;
            leadingBytesOfZero = false;
            break;
        }
        if (leadingBytesOfZero && bytes[15] == 1) {
            return LOOPBACK4;
        }
        if (leadingBytesOfZero && bytes[15] == 0) {
            return ANY4;
        }
        int coercedHash = 0;
        try {
            coercedHash = Hash.hash32(InetAddresses.getEmbeddedIPv4ClientAddress((Inet6Address)ip).getAddress());
        }
        catch (IllegalArgumentException acceptable) {
            coercedHash = Hash.hash32(ip.getAddress(), 0, 8);
        }
        if ((coercedHash |= 0xE0000000) == -1) {
            coercedHash = -2;
        }
        return InetAddresses.getInet4Address(Ints.toByteArray(coercedHash));
    }

    public static int coerceToInteger(InetAddress ip) {
        return ByteStreams.newDataInput(InetAddresses.getCoercedIPv4Address(ip).getAddress()).readInt();
    }

    public static Inet4Address fromInteger(int address) {
        return InetAddresses.getInet4Address(Ints.toByteArray(address));
    }

    public static InetAddress fromLittleEndianByteArray(byte[] addr) throws UnknownHostException {
        byte[] reversed = new byte[addr.length];
        for (int i = 0; i < addr.length; ++i) {
            reversed[i] = addr[addr.length - i - 1];
        }
        return InetAddress.getByAddress(reversed);
    }

    static /* synthetic */ Inet4Address access$000() {
        return ANY4;
    }
}

