(ns antistock.zookeeper.leader
  (:require [antistock.zookeeper.core :refer :all]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [schema.core :as s]
            [zookeeper.util :refer [sort-sequential-nodes]])
  (:import antistock.zookeeper.core.Zookeeper))

(s/defn ^:always-validate node-from-path
  "Return the node from `path`."
  [path :- s/Str]
  (last (str/split (str path) #"/")))

(s/defn ^:always-validate root-from-path
  "Return the root from `path`."
  [path :- s/Str]
  (let [parts (butlast (str/split (str path) #"/"))
        parts (remove str/blank? parts)]
    (if-not (empty? parts)
      (str "/" (str/join "/" parts)))))

(declare elect-leader)

(s/defn ^:always-validate watch-predecessor
  "Watch the predecessor."
  [zk :- Zookeeper path :- s/Str pred :- s/Str leader :- s/Str
   lead-fn :- s/Any event :- s/Any]
  (let [me (node-from-path path)]
    (if (and (= (:event-type event) :NodeDeleted)
             (= (node-from-path (:path event)) leader))
      (do (log/info "I am the new leader!")
          (lead-fn zk path))
      (if-not (exists
               zk (str (root-from-path path)  "/" pred)
               {:watcher #(watch-predecessor zk path pred leader lead-fn %)})
        (elect-leader zk path lead-fn)))))

(s/defn ^:always-validate predecessor
  "Return all nodes in `coll` before `me`."
  [me :- s/Str coll :- [s/Str]]
  (ffirst (filter #(= (second %) me) (partition 2 1 coll))))

(s/defn ^:always-validate elect-leader
  "Elect a leader in the Zookeeper group."
  [zk :- Zookeeper path :- s/Str lead-fn :- s/Any]
  (let [me (node-from-path path)
        members (sort-sequential-nodes (children zk (root-from-path path)))
        leader (first members)]
    (if (= me leader)
      (do (log/infof "I am %s and I am the leader!" me)
          (lead-fn zk path))
      (let [pred (predecessor me members)]
        (log/infof "I am %s and my predecessor is %s." me pred)
        (if-not (exists
                 zk (str (root-from-path path)  "/" pred)
                 {:watcher #(watch-predecessor zk path pred leader lead-fn %)})
          (elect-leader zk path lead-fn))))))

(s/defn ^:always-validate setup-group
  "Setup the Zookeeper group."
  [zk :- Zookeeper group :- s/Str]
  (when-not (exists zk group)
    (create-all zk group {:persistent? true})
    (log/infof "Created Zookeeper group %s." group)
    true))

(s/defn ^:always-validate join-group
  "Join the Zookeeper group."
  [zk :- Zookeeper group :- s/Str lead-fn :- s/Any & [opts]]
  (setup-group zk group)
  (let [path (create zk (str group  "/n-") (assoc opts :sequential? true))]
    (log/infof "Joined Zookeeper group %s as %s." group (node-from-path path))
    [path (elect-leader zk path lead-fn)]))
