(ns antistock.zookeeper.core
  (:require [clojure.edn :as edn]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [schema.core :as s]
            [zookeeper :as zk]
            [zookeeper.data :as data])
  (:import org.apache.zookeeper.server.auth.DigestAuthenticationProvider))

(def ^:dynamic *defaults*
  "The default Zookeeper config."
  {:server-name "127.0.0.1"
   :server-port 2181})

(s/defrecord Zookeeper
    [connection :- s/Any
     server-name :- s/Str
     server-port :- s/Int]
  {s/Any s/Any})

(s/defn generate-digest :- s/Str
  "Generate the digest for `username` and `password`."
  [username :- s/Str password :- s/Str]
  (DigestAuthenticationProvider/generateDigest (str username ":" password)))

(s/defn digest-acl
  "Creates an instance of an ACL using the digest scheme."
  [username :- s/Str password :- s/Str & perms]
  (let [digest (generate-digest username password)]
    (apply zk/acl "digest" digest (or perms zk/default-perms))))

(defmulti encode-data
  "Encode data into an byte array."
  (fn [data & [opts]] (:codec opts)))

(defmethod encode-data :default [data & [opts]]
  (if data (data/to-bytes (pr-str data))))

(defmulti decode-data
  "Decode data from an byte array."
  (fn [data & [opts]] (:codec opts)))

(defmethod decode-data :default [data & [opts]]
  (if data (edn/read-string (data/to-string data))))

(s/defn children
  "Return the children at `path`."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (apply zk/children (:connection zookeeper) path
         (apply concat opts)))

(s/defn create
  "Create a node at `path`."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (let [opts (update-in opts [:data] encode-data)]
    (apply zk/create (:connection zookeeper) path
           (apply concat opts))))

(s/defn delete
  "Delete the given node, if it exists."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (apply zk/delete (:connection zookeeper) path
         (apply concat opts)))

(s/defn delete-all
  "Delete all nodes in `path`."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (apply zk/delete-all (:connection zookeeper) path
         (apply concat opts)))

(s/defn exists
  "Return true if `node` exists, otherwise false."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (apply zk/exists (:connection zookeeper) path
         (apply concat opts)))

(s/defn create-all
  "Create all node in `path`."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (let [opts (update-in opts [:data] encode-data)]
    (apply zk/create-all (:connection zookeeper) path
           (apply concat opts))))

(s/defn data
  "Return the data at `path`."
  [zookeeper :- Zookeeper path :- s/Str & [opts]]
  (-> (apply zk/data (:connection zookeeper) path
             (apply concat opts))
      (update-in [:data] decode-data)))

(s/defn set-data
  "Sets the value of the data field of the given node."
  [zookeeper :- Zookeeper path :- s/Str data :- s/Any version :- s/Num & [opts]]
  (let [data (encode-data data)]
    (apply zk/set-data (:connection zookeeper) path data
           version (apply concat opts))))

(s/defn connect
  "Connect to zookeeper."
  [zookeeper :- Zookeeper]
  (let [{:keys [server-name server-port watcher]} zookeeper
        connection (zk/connect
                    (str server-name
                         (if server-port
                           (str ":" server-port)))
                    :watcher watcher)]
    (when (:username zookeeper)
      (zk/add-auth-info
       connection "digest"
       (str (:username zookeeper) ":"
            (:password zookeeper))))
    connection))

(s/defn start-zookeeper :- Zookeeper
  "Start the Zookeeper."
  [zookeeper :- Zookeeper]
  (if (:connection zookeeper)
    zookeeper
    (let [connection (connect zookeeper)]
      (log/infof "Zookeeper connection to %s on port %s established."
                 (:server-name zookeeper)
                 (:server-port zookeeper))
      (assoc zookeeper :connection connection))))

(s/defn stop-zookeeper :- Zookeeper
  "Stop the Zookeeper."
  [zookeeper :- Zookeeper]
  (when-let [connection (:connection zookeeper)]
    (zk/close connection)
    (log/infof "Zookeeper connection to %s on port %s closed."
               (:server-name zookeeper)
               (:server-port zookeeper)))
  (assoc zookeeper :connection nil))

(extend-protocol component/Lifecycle
  Zookeeper
  (start [zookeeper]
    (start-zookeeper zookeeper))
  (stop [zookeeper]
    (stop-zookeeper zookeeper)))

(s/defn zookeeper :- Zookeeper
  "Make a new Zookeeper component."
  [config]
  (map->Zookeeper (merge *defaults* config)))

(defmacro with-zookeeper
  "Create a new zookeeper, bind the started zookeeper to
  `zookeeper-sym`, evaluate `body` and stop the zookeeper again."
  [[zookeeper-sym config] & body]
  `(let [zookeeper# (component/start (zookeeper ~config))
         ~zookeeper-sym zookeeper#]
     (try ~@body
          (finally (component/stop zookeeper#)))))
