(ns antistock.zookeeper.config
  (:refer-clojure :exclude [read])
  (:require [antistock.zookeeper.core :as zookeeper]
            [clojure.set :as set]
            [schema.core :as s]
            [zookeeper :as zk])
  (:import antistock.zookeeper.core.Zookeeper
           java.util.Map))

(s/defn read :- (s/maybe Map)
  "Read the config under `root` from `zookeeper`."
  [zookeeper :- Zookeeper root :- String]
  (let [children (zookeeper/children zookeeper root)]
    (cond
      (sequential? children)
      (reduce
       (fn [config child]
         (let [path (str root "/" child)
               children (zookeeper/children zookeeper path)]
           (->> (cond
                  (empty? children)
                  (:data (zookeeper/data zookeeper path))
                  :else
                  (read zookeeper path))
                (assoc config (keyword child)))))
       {} children)
      (zookeeper/exists zookeeper root)
      (:data (zookeeper/data zookeeper root)))))

(s/defn write :- Map
  "Write the `config` under `root` in `zookeeper`."
  [zookeeper :- Zookeeper root :- String config :- Map & [opts]]
  (when-let [children (zookeeper/children zookeeper root)]
    (doseq [k (set/difference (set children) (set (map name (keys config))))]
      (zookeeper/delete-all zookeeper (str root "/" k))))
  (doseq [[k v] config]
    (cond
      (map? v)
      (write zookeeper (str root "/" (name k)) v opts)
      :else
      (zookeeper/create-all
       zookeeper (str root "/" (name k))
       (assoc opts :data v :persistent? true))))
  config)
