(ns antistock.zooconf.main
  "Synchronize the Antistock configuration with ZooKeeper."
  (:gen-class)
  (:require [antistock.config.core :as config]
            [antistock.curator.core :as curator]
            [antistock.util :as util]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [environ.core :refer [env]]
            [inflections.core :refer [pluralize]]
            [no.en.core :refer [format-url parse-url]]
            [schema.core :as s])
  (:import antistock.curator.core.Curator))

(s/defschema Env
  "The schema for the environment."
  {s/Keyword s/Str})

(def default-path
  "The default ZooKeeper path for the configuration."
  "/antistock/config")

(s/defn config :- {s/Keyword s/Any}
  "Return the curator config from `env` and `opts`"
  [env :- Env & [opts]]
  (curator/curator
   (or (some-> opts :zookeeper parse-url)
       (config/zookeeper env))))

(s/defn service-config :- {s/Keyword s/Any}
  "Return the service config from `env`"
  [env :- Env]
  {:api (config/api-system env)
   :worker (config/worker-system env)
   :tweetlogd (config/tweetlogd-system env)})

(keys (:worker (service-config env)))

(s/defn public-url :- s/Str
  "Return the public zookeeper url for `path`."
  [curator :- Curator path :- s/Str]
  (-> (assoc curator :uri path :scheme :zookeeper)
      (dissoc :password)
      (format-url)))

(s/defn synchronize :- {s/Keyword s/Any}
  "Synchronize the `config` in Zookeeper using the `curator` framework."
  [env :- Env path :- s/Str opts]
  (curator/with-curator [curator (config env opts)]
    (log/infof "Synchronizing configuration at %s." (public-url curator path))
    (let [config (service-config env)]
      (doseq [[key value] config
              :let [path (str path "/" (name key))
                    bytes (curator/encode curator value)]]
        (if-not (curator/exists? curator path)
          (curator/create curator path value {:persistent? true})
          (curator/set-data curator path value))
        (log/infof " * %s, %s." path (pluralize (count bytes) "byte"))
        (clojure.pprint/pprint value))
      (log/infof "Successfully synchronized configuration.")
      config)))

(s/defn clean
  "Remove the config from Zookeeper."
  [env :- Env path :- s/Str opts]
  (curator/with-curator [curator (config env opts)]
    (log/infof "Removing configuration from %s." (public-url curator path))
    (when (curator/exists? curator path)
      (curator/delete curator path))
    (log/infof "Successfully removed configuration.")))

(defn -main
  "Synchronize the Antistock configuration with Zookeeper."
  [& args]
  (util/log-banner)
  (log/info "            ⋆⋆⋆ ZOOKEEPER CONFIGURATION SYNC ⋆⋆⋆")
  (log/info "")
  (cli/with-commandline [[opts args] args]
    [[h help "Print this help."]
     [c clean "Clean the configuration in ZooKeeper."]
     [p path "The ZooKeeper synchronization path." :string "ZOOKEEPER-PATH"]
     [n no-schema "Disable schema validation for the configuration."]
     [Z zookeeper "The ZooKeeper URL." :string "ZOOKEEPER-URL"]]
    (if (:help opts)
      (cli/print-help "as configure")
      (let [path (or (:path opts) default-path)]
        (s/set-fn-validation! (or (not (:no-schema opts)) true))
        (if (:clean opts)
          (clean env path opts)
          (synchronize env path opts))))))
