(ns antistock.db.prices
  (:refer-clojure :exclude [distinct group-by update])
  (:require [clj-time.core :refer [now minus years]]
            [clj-time.coerce :refer [to-date-time to-sql-time]]
            [clojure.tools.logging :as log]
            [antistock.yahoo.finance :refer [historical]]
            [antistock.util :refer [indexed]]
            [datumbazo.core :refer :all]))

(deftable prices
  "The stock prices database table."
  (column :id :serial :primary-key? true)
  (column :quote-id :integer :references :quotes.id :not-null? true)
  (column :date :date :not-null? true)
  (column :open :float :not-null? true)
  (column :close :float :not-null? true)
  (column :high :float :not-null? true)
  (column :low :float :not-null? true)
  (column :volume :float :not-null? true)
  (column :adj-close :float :not-null? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(defquery1 stock-market-date
  "Returns the closest date to `date` on which the stock market was open."
  [db date]
  (select db [(as '(max date) :date)]
    (from :prices)
    (where `(<= :date ~(to-sql-time date))))
  :date)

(defquery stock-market-dates
  "Returns the stock market dates between `start` and `end`."
  [db start end]
  (select db [:date]
    (from :prices)
    (where `(and (>= :date ~(to-sql-time start))
                 (<= :date ~(to-sql-time end))))
    (group-by :date)
    (order-by :date))
  :date)

(defn prices-by-quote
  "Returns all prices by `quote`."
  [db quote & opts]
  (apply prices-by-quote-id db (:id quote) opts))

(defquery quote-prices
  "Returns all prices for `quote`."
  [db quote]
  (select db [*]
    (from :prices)
    (where `(= :quote-id ~(:id quote)))
    (order-by :date)))

(defquery prices-at
  "Returns all prices at `date`."
  [db date & {:keys [quotes]}]
  (select db [*]
    (from :prices)
    (where `(and (= :date ~(to-sql-time date))))
    (if-not (empty? quotes)
      (where `(in :quote-id ~(map :id quotes)) :and))
    (order-by :quote-id)))

(defn price-at
  "Returns the price for `quote` at `date`."
  [db quote date]
  (first (prices-at db date :quotes [quote])))

(defquery prices-between
  "Returns all prices that are between `start` and `end`."
  [db start end & {:keys [quotes]}]
  (select db [*]
    (from :prices)
    (where `(and (>= :date ~(to-sql-time start))
                 (<= :date ~(to-sql-time end))))
    (if-not (empty? quotes)
      (where `(in :quote-id ~(map :id quotes)) :and))
    (order-by :date :quote-id)))

(defquery update-daily-return
  "Update the daily-return column for the prices of `quote`."
  [db quote & {:keys [start end]}]
  (update db :prices '((= :daily-return :u.daily-return))
    (from (as (select db [:id (as '(- (/ :adj-close ((lag :adj-close) over (partition by :quote-id order by :date))) 1) :daily-return)]
                (from :prices)
                (where `(= :prices.quote-id ~(:id quote)))) :u))
    (where `(and (= :prices.id :u.id)
                 (= :prices.quote-id ~(:id quote))))
    (if start (where `(>= :date ~(to-sql-time start)) :and))
    (if end (where `(<= :date ~(to-sql-time end)) :and))))

(defquery delete-quote-prices-between
  "Delete all prices of `quote` that are between `start` and `end`."
  [db quote start end]
  (delete db :prices
    (where `(and (= :quote-id ~(:id quote))
                 (>= :date ~(to-sql-time start))
                 (<= :date ~(to-sql-time end))))))

(defquery1 total-return
  "Returns the anual return of `quote` between `start` and `end`."
  [db quote & {:keys [start end]}]
  (let [end (to-sql-time (or end (now)))
        start (to-sql-time (or start (minus end (years 1))))]
    (select db [(as `(- (/ ~(select db [:adj-close]
                              (from :prices)
                              (where `(and (= :quote-id ~(:id quote))
                                           (= :date ~(select db ['(max :date)]
                                                       (from :prices)
                                                       (where `(and (= :quote-id ~(:id quote))
                                                                    (>= :date ~start)
                                                                    (<= :date ~end))))))))
                           ~(select db [:adj-close]
                              (from :prices)
                              (where `(and (= :quote-id ~(:id quote))
                                           (= :date ~(select db ['(min :date)]
                                                       (from :prices)
                                                       (where `(and (= :quote-id ~(:id quote))
                                                                    (>= :date ~start)
                                                                    (<= :date ~end)))))))))
                        1) :total-return)]))
  :total-return)

(defquery1 sharp-ratio-by-quote
  "The standard deviataion of `quote` between `start` and `end`."
  [db quote & {:keys [start end]}]
  (let [end (to-sql-time (or end (now)))
        start (to-sql-time (or start (minus end (years 1))))]
    (select db [(as `(/ ~(select db ['(avg :daily-return)]
                           (from :prices)
                           (where `(and (= :quote-id ~(:id quote))
                                        (>= :date ~start)
                                        (<= :date ~end))))
                        ~(select db ['(stddev_pop :daily-return)]
                           (from :prices)
                           (where `(and (= :quote-id ~(:id quote))
                                        (>= :date ~start)
                                        (<= :date ~end)))))
                    :sharp-ratio)]))
  :sharp-ratio)

(defquery1 standard-deviation-by-quote
  "The standard deviataion of `quote` between `start` and `end`."
  [db quote & {:keys [start end]}]
  (let [end (to-sql-time (or end (now)))
        start (to-sql-time (or start (minus end (years 1))))]
    (select db [(as '(stddev_pop :daily-return) :standard-deviation)]
      (from :prices)
      (where `(and (= :quote-id ~(:id quote))
                   (>= :date ~start)
                   (<= :date ~end)))))
  :standard-deviation)

(defquery total-returns
  "Returns the anual returns of all quotes between `start` and `end`."
  [db & {:keys [start end]}]
  (let [end (or end (now))
        start (to-sql-time (or start (minus end (years 1))))
        end (to-sql-time end)]
    (select db [:start-prices.quote-id
                :start-date
                (as :start-prices.adj-close :start-adj-close)
                (as :start-prices.close :start-close)
                (as :start-prices.daily-return :start-daily-return)
                (as :start-prices.high :start-high)
                (as :start-prices.low :start-low)
                (as :start-prices.open :start-open)
                (as :start-prices.volume :start-volume)
                :end-date
                (as :end-prices.adj-close :end-adj-close)
                (as :end-prices.close :end-close)
                (as :end-prices.daily-return :end-daily-return)
                (as :end-prices.high :end-high)
                (as :end-prices.low :end-low)
                (as :end-prices.open :end-open)
                (as :end-prices.volume :end-volume)
                (as '(- (/ :end-prices.close :start-prices.close) 1) :total-return)]
      (from (as (select db [:prices.* :start-date]
                  (from :prices)
                  (join (as (select db [:quote-id (as '(min :date) :start-date)]
                              (from :prices)
                              (group-by :quote-id))
                            :start-dates)
                        '(on (and (= :prices.quote-id :start-dates.quote-id)
                                  (= :prices.date :start-dates.start-date)))))
                :start-prices)
            (as (select db [:prices.* :end-date]
                  (from :prices)
                  (join (as (select db [:quote-id (as '(max :date) :end-date)]
                              (from :prices)
                              (group-by :quote-id))
                            :end-dates)
                        '(on (and (= :prices.quote-id :end-dates.quote-id)
                                  (= :prices.date :end-dates.end-date)))))
                :end-prices))
      (where '(= :start-prices.quote-id :end-prices.quote-id))
      (order-by :start-prices.quote-id))))

(defn load-prices-for-quote
  "Load stock prices for `quote` into the database."
  [db quote & {:keys [start end]}]
  (let [prices (historical quote :start start :end end)]
    (when-not (empty? prices)
      (let [dates (sort (map :date prices))]
        (with-transaction
          [db db]
          (delete-quote-prices-between db quote (first dates) (last dates))
          (let [rows (insert-prices db prices)]
            (update-daily-return db quote :start start :end end)
            rows))))))

(defn load-prices-for-quotes
  "Load stock prices for `quotes` into the database."
  [db quotes & {:keys [batch start end]}]
  (let [total (count quotes)]
    (doseq [[n quote] (indexed quotes)]
      (try
        (log/infof "[%s/%s] Loaded %s prices for quote %s." (inc n) total
                   (count (load-prices-for-quote db quote :start start :end end))
                   (:symbol quote))
        (catch Exception e
          (log/warnf "Can't load prices for quote %s: %s."
                     (:symbol quote) (.getMessage e)))))))
