(ns antistock.db.fixtures
  (:gen-class)
  (:require [antistock.config.core :as config]
            [antistock.db.system :refer [with-db]]
            [antistock.util :refer [log-banner]]
            [clojure.tools.logging :refer [infof]]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :refer [with-connection cascade truncate]]
            [datumbazo.fixtures :as fixtures]
            [environ.core :refer [env]]
            [antistock.db.util :as util]))

(def ^:dynamic *tables*
  [:continents
   :countries
   :exchanges
   :sectors
   :industries
   :languages
   :companies
   :quotes
   :users
   :prices
   :links
   :links-quotes
   :twitter.tweets
   :twitter.hash-tags
   :twitter.hash-tags-tweets
   :twitter.links-tweets
   :twitter.tweets-quotes
   :twitter.users
   :twitter.tweets-users
   :wikipedia.pages])

(def ^:dynamic *fixture-path*
  "resources/db/antistock-db/fixtures")

(defn dump-fixtures [db]
  (infof "Dumping fixtures from database.")
  (fixtures/dump-fixtures db *fixture-path* *tables*))

;; TODO: Read fixtures from resources, not from directory.
(defn load-fixtures [db & [fixtures]]
  (infof "Loading fixtures into database.")
  (->> (fixtures/load-fixtures db *fixture-path*)
       (map #(dissoc %1 :records))))

(defn delete-fixtures [db]
  (infof "Deleting fixtures from database.")
  (doseq [table (reverse *tables*)]
    @(truncate db [table] (cascade true))))

(defn show-help []
  (print-help "fixtures [delete|dump|load|]")
  (System/exit 0))

(defn run [db cmd & [opts]]
  (case cmd
    "delete"
    (delete-fixtures db)
    "dump"
    (dump-fixtures db)
    "load"
    (do (load-fixtures db)
        (util/refresh-materialized-views db))
    (show-help)))

(defn -main [& args]
  (with-commandline [[opts [cmd & args]] args]
    [[h help "Print this help."]]
    (when (:help opts) (show-help))
    (log-banner)
    (with-db [db (config/db env)]
      (with-connection [db db]
        (run db cmd)))))

(comment
  (fixtures/dump-fixtures (db) *fixture-path* [:wikipedia.projects])
  (fixtures/fixture-seq *fixture-path*)
  (time (dump-fixtures))
  (time (load-fixtures (db)))
  (delete-fixtures (db)))
