(ns antistock.db.twitter
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.db.util :refer [fulltext]]
            [clj-time.coerce :refer [to-date-time]]
            [clojure.edn :as edn]
            [datumbazo.core :refer :all]))

(deftable twitter-users
  "The Twitter users database table."
  (table :twitter.users)
  (column :id :serial :primary-key? true)
  (column :screen-name :text :not-null? true)
  (column :name :text :not-null? true)
  (column :followers-count :integer :not-null? true :default 0)
  (column :friends-count :integer :not-null? true :default 0)
  (column :retweet-count :integer :not-null? true :default 0)
  (column :statuses-count :integer :not-null? true :default 0)
  (column :listed-count :integer :not-null? true :default 0)
  (column :verified :boolean :not-null? true :default false)
  (column :possibly-sensitive :boolean :not-null? true :default false)
  (column :default-profile-image :boolean)
  (column :description :text)
  (column :lang :varchar :length 2)
  (column :location :text)
  (column :profile-image-url :text)
  (column :time-zone :text)
  (column :url :text)
  (column :utc-offset :integer)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(deftable twitter-tweets
  "The Twitter tweets database table."
  (table :twitter.tweets)
  (column :id :serial :primary-key? true)
  (column :user-id :integer :not-null? true :references :twitter.users/id)
  (column :retweeted :boolean :not-null? true :default false)
  (column :retweet-count :integer)
  (column :text :text :not-null? true)
  (column :sentiment :integer)
  (column :favorited :boolean :not-null? true :default false)
  (column :in-reply-to-status-id :integer)
  (column :in-reply-to-user-id :integer)
  (column :in-reply-to-screen-name :text)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(deftable tweets-quotes
  "The join table between Twitter tweets and stock quotes."
  (table :twitter.tweets-quotes)
  (column :tweet-id :integer :not-null? true :references :twitter.tweets/id)
  (column :quote-id :integer :not-null? true :references :quotes/id)
  (primary-key :tweet-id :quote-id))

(deftable links-tweets
  "The join table between links and tweets."
  (table :twitter.links-tweets)
  (column :link-id :integer :not-null? true :references :links/id)
  (column :tweet-id :integer :not-null? true :references :tweets/id)
  (primary-key :link-id :tweet-id))

(deftable tweets-users
  "The join table between tweets and users."
  (table :twitter.tweets-users)
  (column :user-id :integer :not-null? true :references :users/id)
  (column :tweet-id :integer :not-null? true :references :tweets/id)
  (primary-key :user-id :tweet-id))

(deftable twitter-hash-tags
  "The Twitter hash tags table."
  (table :twitter.hash-tags)
  (column :id :bigserial :primary-key? true)
  (column :name :citext :not-null? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()")
  (primary-key :name))

(deftable hash-tags-tweets
  "The join table between hash tags and tweets."
  (table :twitter.hash-tags-tweets)
  (column :hash-tag-id :integer :not-null? true :references :hash-tags/id)
  (column :tweet-id :integer :not-null? true :references :tweets/id)
  (primary-key :hash-tag-id :tweet-id))

(defquery tweets-by-quote
  "Returns the tweets for `quote`."
  [db quote & {:as opts}]
  (select db [*]
    (from :twitter.tweets)
    (join :twitter.tweets-quotes.tweet-id :twitter.tweets.id)
    (where `(= :twitter.tweets-quotes.quote-id ~(:id quote)))
    (paginate (:page opts) (:per-page opts))))

(defquery twitter-tweets
  "Return the Twitter statuses."
  [db & [opts]]
  (select db [*]
    (from :twitter.tweets)
    (order-by (desc :twitter.tweets.created-at))
    (fulltext (:query opts) :twitter.tweets.text)
    (paginate (:page opts) (:per-page opts))))

(defquery twitter-users
  "Return the Twitter users."
  [db & [opts]]
  (select db [*]
    (from :twitter.users)
    (order-by (desc :twitter.users.updated-at))
    (fulltext (:query opts) :twitter.users.name :twitter.users.screen-name :twitter.users.description)
    (paginate (:page opts) (:per-page opts))))

(defn create-twitter-import-table
  "Create the Twitter statuses import table."
  [db]
  @(create-table db :twitter.import
     (column :data :json)))

(defn drop-twitter-import-table
  "Drop the Twitter statuses import table."
  [db & {:as opts}]
  @(drop-table db [:twitter.import]
     (if-exists (:if-exists opts))))
