(ns antistock.db.companies
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.db.util :as util]
            [clj-time.coerce :refer [to-date-time]]
            [datumbazo.core :refer :all]))

(deftable companies
  "The companies database table."
  (column :id :serial :primary-key? true)
  (column :country-id :integer :references :countries/id)
  (column :industry-id :integer :references :industries/id)
  (column :wikipedia-page-id :integer :references :wikipedia.pages/id)
  (column :name :citext :not-null? true :unique? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(defquery companies
  "Returns the companies."
  [db & [opts]]
  (select db [:companies.*]
    (from :companies)
    (util/fulltext (:query opts) :companies.name)
    (paginate (:page opts) (:per-page opts))
    (order-by :name)))

(defquery companies-by-updated-at-asc
  "Returns the companies by updated-at column in asc order."
  [db] (compose (companies* db)
                (order-by (asc :updated-at))))

(defn select-companies
  "Return the SQL select statement for companies."
  [db & [opts]]
  (select db [:companies.id
              :companies.name
              :companies.created-at
              :companies.updated-at
              (util/json-embed-country)
              (util/json-embed-industry)
              (util/json-embed-sector)]
    (from :companies)
    (join :countries.id :companies.country-id :type :left)
    (join :industries.id :companies.industry-id :type :left)
    (join :sectors.id :companies.sector-id :type :left)
    (util/fulltext (:query opts) :companies.name)
    (paginate (:page opts) (:per-page opts))
    (order-by :companies.name)))

(defn all
  "Return the companies."
  [db & [opts]]
  @(select-companies db opts))

(defn by-id
  "Return the company by `id`."
  [db id & [opts]]
  (->> (compose
        (select-companies db opts)
        (where `(= :companies.id ~id) :and))
       deref first))
