(ns antistock.db.users
  (:refer-clojure :exclude [distinct group-by update])
  (:require [clojure.java.jdbc :as jdbc]
            [datumbazo.core :refer :all]))

(deftable users
  "The users database table."
  (column :id :serial :primary-key? true)
  (column :name :citext :not-null? true :unique? true)
  (column :email :citext :not-null? true :unique? true)
  (column :crypted-password :citext :not-null? true :unique? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(defn encrypt-password
  "Encrypt `password` with the Blowfish cipher on the database."
  [db password] (:crypt (first @(select db [`(crypt ~password (gen_salt "bf" 10))]))))

(defn login
  "Try to login `user` with `password`."
  [db user password]
  (run1 (select db [:*]
          (from :users)
          (where `(and (= :crypted-password
                          ~(select db [`(crypt ~password :crypted-password)]))
                       (or (= :id ~(:id user))
                           (= :email ~(:email user))
                           (= :name ~(:name user))))))))

(defn roman [db]
  (user-by-id db 1))


(defquery1 authenticate-password
  "Try to authenticate `user` with a password."
  [db user]
  (let [login (first (remove nil? (map (or user {}) [:login :username :email])))]
    (compose (users* db)
             (where `(and (or (= :username ~login)
                              (= :email ~login))
                          (= :crypted-password  (crypt ~(:password user) :crypted-password)))))))

(defn authenticate
  "Rerturns a friend compatible authentication map."
  [db user]
  (if-let [user (authenticate-password db user)]
    (->> ;; (roles-by-user db user)
         [:user]
         (map #(keyword (:name %1)))
         (set)
         (assoc user :roles))))
