(ns antistock.yahoo.finance
  (:require [clj-time.core :refer [date-time now year month day]]
            [clj-time.coerce :refer [to-date-time]]
            [clojure.data.csv :refer [read-csv]]
            [clojure.string :refer [join]]
            [clojure.tools.logging :refer [debugf]]
            [no.en.core :refer [parse-double parse-long parse-percent]]
            [antistock.util :refer :all]))

(def ^:dynamic *formats* {})

(defn- parse-current-record
  "Parse the current `record`."
  [record]
  (reduce
   #(assoc %1
           (:id %2)
           ((:parse-fn %2) (get %1 (:id %2))))
   record (vals *formats*)))

(defn- parse-historical-record
  "Parse the historical `record`."
  [record]
  (-> record
    (update-in [:adj-close] parse-double)
    (update-in [:close] parse-double)
    (update-in [:date] to-date-time)
    (update-in [:high] parse-double)
    (update-in [:low] parse-double)
    (update-in [:open] parse-double)
    (update-in [:volume] parse-long)))

(defn current
  "Returns a seq of real time stock data for `quotes`."
  [quotes & {:keys [formats batch]}]
  (let [batch (or batch 200)]
    (mapcat
     (fn [quotes]
       (debugf "Fetching current stock data for %s.\n"
               (join ", " (map :symbol quotes)))
       (let [formats (or formats (keys *formats*))
             symbols (zipmap (map :symbol quotes) quotes)]
         (map #(let [r (parse-current-record (zipmap formats %1))]
                 (assoc r :quote-id (:id (get symbols (:symbol r)))))
              (->  (http {:conn-timeout (* 1000 5)
                          :method :get
                          :socket-timeout (* 1000 5)
                          :url "http://finance.yahoo.com/d/quotes.csv"
                          :query-params
                          {:s (join "+" (map :symbol quotes))
                           :f (join "" (map (comp :format *formats*)
                                            formats))}})
                   :body read-csv))))
     (partition batch batch [] quotes))))

(defn historical
  "Returns a seq of stock data for `quote` between `start` and `end`."
  [quote & {:keys [frequency start end]}]
  (let [start (or (to-date-time start) (beginning-of-year (now)))
        end (or (to-date-time end) (now))]
    (debugf "Fetching historical stock data for %s between %s and %s.\n"
            (:symbol quote) start end)
    (->> (http {:conn-timeout (* 1000 5)
                :method :get
                :socket-timeout (* 1000 5)
                :url "http://ichart.finance.yahoo.com/table.csv"
                :query-params
                {:a (dec (month start))
                 :b (day start)
                 :c (year start)
                 :d (dec (month end))
                 :e (day end)
                 :f (year end)
                 :s (:symbol quote)
                 :g (case frequency
                      :monthly ""
                      :weekly "w"
                      "d")
                 :ignore ".csv"}})
         :body csv-seq->map
         (map #(assoc (parse-historical-record %1) :quote-id (:id quote))))))

(defmacro defformat [id description format & [parse-fn]]
  `(alter-var-root
    #'*formats* assoc ~(keyword id)
    {:id ~(keyword id)
     :format ~(name format)
     :description ~description
     :parse-fn ~(or parse-fn identity)}))

(defformat :1-year-target-price
  "1 Year Target Price" "t8" parse-double)

(defformat :200-day-moving-average
  "200 Day Moving Average" "m4" parse-double)

(defformat :50-day-moving-average
  "50 Day Moving Average" "m3" parse-double)

(defformat :52-week-high
  "52 Week High" "k" parse-double)

(defformat :52-week-low
  "52 Week Low" "j" parse-double)

;; (defformat :52-week-range
;;   "52 Week Range" "w")

(defformat :after-hours-change-real-time
  "After Hours Change (Real-time)" "c8" parse-double)

;; (defformat :annualized-gain
;;   "Annualized Gain" "g3")

(defformat :ask
  "Ask Price" "a" parse-double)

(defformat :ask-real-time
  "Ask Price (Real-time)" "b2" parse-double)

(defformat :ask-size
  "Ask Size" "a5" parse-long)

(defformat :avg-daily-vol
  "Average Daily Volume" "a2" parse-long)

(defformat :bid
  "Bid Price" "b" parse-double)

(defformat :bid-real-time
  "Bid Price (Real-time)" "b3" parse-double)

(defformat :bid-size
  "Bid Size" "b6" parse-long)

(defformat :book-value
  "Book Value" "b4" parse-double)

(defformat :change
  "Change" "c1" parse-double)

(defformat :change-from-200-day-moving-average
  "Change From 200-day Moving Average" "m5" parse-percent)

(defformat :change-from-50-day-moving-average
  "Change From 50-day Moving Average" "m7" parse-percent)

(defformat :change-from-52-week-high
  "Change From 52-week High" "k4" parse-percent)

(defformat :change-from-52-week-low
  "Change From 52-week Low" "j5" parse-percent)

(defformat :change-in-percent
  "Change in Percent" "p2" parse-percent)

;; (defformat :change-percent
;;   "Change Percent" "c")

;; (defformat :change-percent-real-time
;;   "Change Percent (Real-time)" "k2")

(defformat :change-real-time
  "Change (Real-time)" "c6" parse-double)

;; (defformat :commision
;;   "Commision" "c3")

(defformat :day-high
  "Day High" "h" parse-double)

(defformat :day-low
  "Day Low" "g" parse-double)

;; (defformat :day-range
;;   "Day Range" "m")

;; (defformat :day-range-real-time
;;   "Day Range (Real-time)" "m2")

;; (defformat :day-value-change
;;   "Day Value Change" "w1")

;; (defformat :day-value-change-real-time
;;   "Day Value Change (Real-time)" "w4")

(defformat :dividend-pay-date
  "Dividend Pay Date" "r1" to-date-time)

(defformat :dividend-share
  "Dividend Share" "d" parse-double)

(defformat :dividend-yield
  "Dividend Yield" "y" parse-double)

(defformat :earnings-share
  "Earnings Share" "e" parse-double)

(defformat :ebitda
  "EBITDA" "j4" parse-double)

(defformat :eps-estimate-current-year
  "EPS Estimate Current Year" "e7" parse-double)

(defformat :eps-estimate-next-quarter
  "EPS Estimate Next Quarter" "e9" parse-double)

(defformat :eps-estimate-next-year
  "EPS Estimate Next Year" "e8" parse-double)

(defformat :ex-dividend-date
  "Ex-Dividend Date" "q" to-date-time)

(defformat :float-shares
  "Float Shares" "f6" parse-double)

(defformat :high-limit
  "High Limit" "l2" parse-double)

(defformat :holdings-gain
  "Holdings Gain" "g4" parse-double)

;; (defformat :holdings-gain-percent
;;   "Holdings Gain Percent" "g1")

;; (defformat :holdings-gain-percent-real-time
;;   "Holdings Gain Percent (Real-time)" "g5")

(defformat :holdings-gain-real-time
  "Holdings Gain (Real-time)" "g6" parse-double)

;; (defformat :holdings-value
;;   "Holdings Value" "v1")

(defformat :holdings-value-real-time
  "Holdings Value (Real-time)" "v7" parse-double)

(defformat :last-trade-date
  "Last Trade Date" "d1" parse-yahoo-date)

(defformat :last-trade-price-only
  "Last Trade (Price Only)" "l1" parse-double)

;; (defformat :last-trade-real-time-with-time
;;   "Last Trade (Real-time) With Time" "k1")

(defformat :last-trade-size
  "Last Trade Size" "k3" parse-long)

(defformat :last-trade-time
  "Last Trade Time" "t1" to-date-time)

;; (defformat :last-trade-with-time
;;   "Last Trade With Time" "l")

(defformat :low-limit
  "Low Limit" "l3" parse-double)

(defformat :market-cap
  "Market Capitalization" "j1" parse-double)

(defformat :market-cap-real-time
  "Market Capitalization (Real-time)" "j3" parse-double)

(defformat :more-info
  "More Info" "i")

(defformat :name
  "Name" "n")

(defformat :notes
  "Notes" "n4")

(defformat :open
  "Open Price" "o" parse-double)

(defformat :order-book-real-time
  "Order Book (Real-time)" "i5" parse-double)

(defformat :p-e-ratio
  "P/E Ratio" "r" parse-double)

(defformat :p-e-ratio-real-time
  "P/E Ratio (Real-time)" "r2" parse-double)

(defformat :peg-ratio
  "PEG Ratio" "r5" parse-double)

(defformat :percent-change-from-200-day-moving-average
  "Percent Change From 200-day Moving Average" "m6" parse-percent)

(defformat :percent-change-from-50-day-moving-average
  "Percent Change From 50-day Moving Average" "m8" parse-percent)

(defformat :percent-change-from-52-week-high
  "Percent Change From 52-week High" "k5" parse-percent)

(defformat :percent-change-from-52-week-low
  "Percent Change From 52-week Low" "j6" parse-percent)

(defformat :previous-close
  "Previous Close" "p" parse-double)

(defformat :price-book
  "Price Book" "p6" parse-double)

(defformat :price-eps-estimate-current-year
  "Price/EPS Estimate Current Year" "r6" parse-double)

(defformat :price-eps-estimate-next-year
  "Price/EPS Estimate Next Year" "r7" parse-double)

(defformat :price-paid
  "Price Paid" "p1" parse-double)

(defformat :price-sales
  "Price Sales" "p5" parse-double)

;; (defformat :shares-owned
;;   "Shares Owned" "s1")

(defformat :short-ratio
  "Short Ratio" "s7" parse-double)

;; (defformat :stock-exchange
;;   "Stock Exchange" "x")

(defformat :symbol
  "Symbol" "s")

;; (defformat :ticker-trend
;;   "Ticker Trend" "t7")

(defformat :trade-date
  "Trade Date" "d2" parse-yahoo-date)

(defformat :volume
  "Volume" "v" parse-double)
